# Quiz App Admin Panel Backend (PHP)

## Overview
This backend provides RESTful APIs for the Quiz App Admin Panel. It is built in PHP and uses a custom token-based authentication system for a single admin user.

## Setup
1. Import `snaplimit_quiz_app.sql` into your MySQL database.
2. Update `backend/config.php` with your database credentials.
3. Place the `backend/` folder on your PHP server (e.g., XAMPP, WAMP, LAMP).
4. Make sure `.htaccess` is enabled for routing.

## Authentication
- **Login:** `POST /backend/auth.php?action=login`  
  Body: `{ "username": "admin", "password": "yourpassword" }`
- **Logout:** `POST /backend/auth.php?action=logout`  
  Header: `Authorization: <token>`
- All other endpoints require the `Authorization` header with the token.

## Endpoints
- **Users:** `/backend/users.php` (GET, POST, PUT, DELETE)
- **Ads:** `/backend/ads.php` (GET, POST, PUT, DELETE)
- **Rewards:** `/backend/rewards.php` (GET, POST, PUT, DELETE)
- **Settings:** `/backend/settings.php` (GET, PUT)
- **Tasks:** `/backend/tasks.php` (GET, POST, PUT, DELETE)

## Database Notes
- Admin credentials are in the `admin_users` table. Passwords are hashed.
- The custom token is stored in the `remember_token` field of `admin_users`.

## Example Usage
- Login to get a token, then use that token in the `Authorization` header for all requests.

# API Documentation

All endpoints require the `Authorization` header with the latest token, except for login. After every successful API call, use the new token from the response for the next request.

**Base URL:** `https://snaplimit.com/admin/backend/`

---

## Authentication

### Login
- **Endpoint:** `POST https://snaplimit.com/admin/backend/auth.php?action=login`
- **Body:**
```json
{
  "username": "admin",
  "password": "yourpassword"
}
```
- **Curl:**
```sh
curl -X POST \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"yourpassword"}' \
  https://snaplimit.com/admin/backend/auth.php?action=login
```
- **Response:**
```json
{ "token": "<token>" }
```

### Logout
- **Endpoint:** `POST /backend/auth.php?action=logout`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X POST \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/auth.php?action=logout
```
- **Response:**
```json
{ "success": true }
```

---

## Users

### List Users
- **Endpoint:** `GET /backend/users.php`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X GET \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/users.php
```
- **Response:**
```json
{ "data": [ ...users... ], "token": "<new_token>" }
```

### Create User
- **Endpoint:** `POST /backend/users.php`
- **Headers:** `Authorization: <token>`
- **Body:**
```json
{
  "email": "user@example.com",
  "device_id": "device123",
  "coins": 100,
  "refercode": "ABC123",
  "referred_by": null,
  "is_blocked": 0
}
```
- **Curl:**
```sh
curl -X POST \
  -H "Authorization: <token>" \
  -H "Content-Type: application/json" \
  -d '{"email":"user@example.com","device_id":"device123","coins":100,"refercode":"ABC123","referred_by":null,"is_blocked":0}' \
  https://snaplimit.com/admin/backend/users.php
```
- **Response:**
```json
{ "success": true, "id": 1, "token": "<new_token>" }
```

### Update User
- **Endpoint:** `PUT /backend/users.php?id=<user_id>`
- **Headers:** `Authorization: <token>`
- **Body (x-www-form-urlencoded):**
```
email=user2@example.com&coins=200
```
- **Curl:**
```sh
curl -X PUT \
  -H "Authorization: <token>" \
  -d "email=user2@example.com&coins=200" \
  https://snaplimit.com/admin/backend/users.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

### Delete User
- **Endpoint:** `DELETE /backend/users.php?id=<user_id>`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X DELETE \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/users.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

---

## Ads

### Get Ads Settings & Controls
- **Endpoint:** `GET /backend/ads.php`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X GET \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/ads.php
```
- **Response:**
```json
{
  "ad_settings": { ... },
  "ad_controls": [ ... ],
  "token": "<new_token>"
}
```

### Update Ads Settings & Controls
- **Endpoint:** `PUT /backend/ads.php`
- **Headers:** `Authorization: <token>`
- **Body (x-www-form-urlencoded):**
```
app_id=...&interstitial_ad_unit=...&ad_controls[0][id]=1&ad_controls[0][interstitial_ad]=1
```
- **Curl:**
```sh
curl -X PUT \
  -H "Authorization: <token>" \
  -d "app_id=...&interstitial_ad_unit=..." \
  https://snaplimit.com/admin/backend/ads.php
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

---

## Rewards

### List Rewards
- **Endpoint:** `GET /backend/rewards.php`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X GET \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/rewards.php
```
- **Response:**
```json
{ "data": [ ...giftcodes... ], "token": "<new_token>" }
```

### Create Reward
- **Endpoint:** `POST /backend/rewards.php`
- **Headers:** `Authorization: <token>`
- **Body:**
```json
{
  "category_id": 1,
  "coins": 100,
  "code": "REWARD123",
  "pin": null,
  "is_used": 0,
  "assigned_to": null
}
```
- **Curl:**
```sh
curl -X POST \
  -H "Authorization: <token>" \
  -H "Content-Type: application/json" \
  -d '{"category_id":1,"coins":100,"code":"REWARD123","pin":null,"is_used":0,"assigned_to":null}' \
  https://snaplimit.com/admin/backend/rewards.php
```
- **Response:**
```json
{ "success": true, "id": 1, "token": "<new_token>" }
```

### Update Reward
- **Endpoint:** `PUT /backend/rewards.php?id=<reward_id>`
- **Headers:** `Authorization: <token>`
- **Body (x-www-form-urlencoded):**
```
coins=200&is_used=1
```
- **Curl:**
```sh
curl -X PUT \
  -H "Authorization: <token>" \
  -d "coins=200&is_used=1" \
  https://snaplimit.com/admin/backend/rewards.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

### Delete Reward
- **Endpoint:** `DELETE /backend/rewards.php?id=<reward_id>`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X DELETE \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/rewards.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

---

## Settings

### Get App Settings
- **Endpoint:** `GET /backend/settings.php`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X GET \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/settings.php
```
- **Response:**
```json
{ "data": [ ... ], "token": "<new_token>" }
```

### Update App Settings
- **Endpoint:** `PUT /backend/settings.php`
- **Headers:** `Authorization: <token>`
- **Body (x-www-form-urlencoded):**
```
app_name=QuizApp&maintenance_mode=0
```
- **Curl:**
```sh
curl -X PUT \
  -H "Authorization: <token>" \
  -d "app_name=QuizApp&maintenance_mode=0" \
  https://snaplimit.com/admin/backend/settings.php
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

---

## Tasks

### List Tasks
- **Endpoint:** `GET /backend/tasks.php`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X GET \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/tasks.php
```
- **Response:**
```json
{ "data": [ ...tasks... ], "token": "<new_token>" }
```

### Create Task
- **Endpoint:** `POST /backend/tasks.php`
- **Headers:** `Authorization: <token>`
- **Body:**
```json
{
  "task_type": "daily",
  "reward_coins": 50,
  "is_enabled": 1
}
```
- **Curl:**
```sh
curl -X POST \
  -H "Authorization: <token>" \
  -H "Content-Type: application/json" \
  -d '{"task_type":"daily","reward_coins":50,"is_enabled":1}' \
  https://snaplimit.com/admin/backend/tasks.php
```
- **Response:**
```json
{ "success": true, "id": 1, "token": "<new_token>" }
```

### Update Task
- **Endpoint:** `PUT /backend/tasks.php?id=<task_id>`
- **Headers:** `Authorization: <token>`
- **Body (x-www-form-urlencoded):**
```
task_type=weekly&reward_coins=100
```
- **Curl:**
```sh
curl -X PUT \
  -H "Authorization: <token>" \
  -d "task_type=weekly&reward_coins=100" \
  https://snaplimit.com/admin/backend/tasks.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

### Delete Task
- **Endpoint:** `DELETE /backend/tasks.php?id=<task_id>`
- **Headers:** `Authorization: <token>`
- **Curl:**
```sh
curl -X DELETE \
  -H "Authorization: <token>" \
  https://snaplimit.com/admin/backend/tasks.php?id=1
```
- **Response:**
```json
{ "success": true, "token": "<new_token>" }
```

---

## Notes
- Always use the latest token from the previous response for the next request.
- All endpoints return a new token after each successful call (except login/logout).
- All requests (except login) require the `Authorization` header.
- Data for PUT requests should be sent as `x-www-form-urlencoded`.

---
**Note:** This backend is for a single admin system and does not use JWT. All tokens are custom-generated and stored in the database.

---

## Redeem Management Module
This module allows admins to manage redeem categories, sub-categories, redemption history, and gift codes.

### 1. Redeem Categories
**Endpoint:** `backend/redeem_categories.php`
- **GET:** List all categories
- **POST:** Add a category (fields: `category_name`, `category_image_url` or `category_image_file`)
- **POST + _method=PUT:** Edit a category (fields: `id`, `category_name`, `category_image_url` or `category_image_file`)
- **DELETE:** Delete a category (`id` as GET param)
- **Image Upload:** Images are stored in `uploads/redeem_categories/`

### 2. Redeem Sub-Categories
**Endpoint:** `backend/redeem_subcategories.php`
- **GET:** List all sub-categories (with parent category name)
- **POST:** Add a sub-category (fields: `category_id`, `sub_category_name`, `sub_category_image_url` or `sub_category_image_file`, `coins`, `cash`, `is_enabled`)
- **POST + _method=PUT:** Edit a sub-category (fields: `id`, ...same as above)
- **DELETE:** Delete a sub-category (`id` as GET param)
- **Image Upload:** Images are stored in `uploads/redeem_subcategories/`

### 3. Redemption History
**Endpoint:** `backend/redeem_history.php`
- **GET:** List all redemption records (filters: `status`, `user`, `category`, `date`)
- **GET + id:** Get details for a specific record
- **POST:** Update status (fields: `id`, `status`, `rejection_reason` if rejected)
- **No DELETE**

### 4. Gift Codes
**Endpoint:** `backend/rewards.php`
- **GET:** List all gift codes
- **POST:** Add a gift code (fields: `category_id`, `coins`, `code`, `pin`, `is_used`, `assigned_to`)
- **POST + _method=PUT:** Edit a gift code (fields: `id`, ...same as above)
- **DELETE:** Delete a gift code (`id` as GET param)

---

## Common Troubleshooting
- **Unauthorized error:** Log in again as admin. All API calls require a valid session.
- **No data shown:** Ensure you have created categories before adding sub-categories. Check your database for existing records.
- **Image not uploading:** Ensure `uploads/redeem_categories/` and `uploads/redeem_subcategories/` are writable by the server.

---

## Other Endpoints
- `backend/users.php` — User management
- `backend/tasks.php` — Task management
- `backend/settings.php` — App settings
- `backend/send_notification.php` — Push notifications

---

## Contact
For support, contact the SnapLimit Quiz App admin developer. 

---

## Promo Codes

### Overview
Promo code management is handled via a dedicated backend API endpoint. The admin UI (`pages/promo_codes.php`) communicates with this API using HTTP requests for all CRUD operations and usage stats. This centralizes business logic and ensures consistency across clients.

### Endpoints
- **List Promo Codes**
  - **Endpoint:** `GET /backend/promo_codes.php?action=list`
  - **Response:**
    ```json
    { "promos": [ { "id": 1, "code": "SUMMER25", "usage_limit": 100, "expiry_date": "2025-06-02 12:00:00", "used": 5 }, ... ] }
    ```

- **Add Promo Code**
  - **Endpoint:** `POST /backend/promo_codes.php`
  - **Body (x-www-form-urlencoded):**
    - `action=add`
    - `code=...`
    - `usage_limit=...`
    - `expiry_date=...`
  - **Response:** `{ "success": true }`

- **Edit Promo Code**
  - **Endpoint:** `POST /backend/promo_codes.php`
  - **Body (x-www-form-urlencoded):**
    - `action=edit`
    - `promo_id=...`
    - `code=...`
    - `usage_limit=...`
    - `expiry_date=...`
  - **Response:** `{ "success": true }`

- **Delete Promo Code**
  - **Endpoint:** `POST /backend/promo_codes.php`
  - **Body (x-www-form-urlencoded):**
    - `action=delete`
    - `promo_id=...`
  - **Response:** `{ "success": true }`

- **Promo Code Usage**
  - **Endpoint:** `GET /backend/promo_codes.php?action=usage&promo_id=...`
  - **Response:**
    ```json
    { "success": true, "usage": [ { "user_id": 1, "email": "user@example.com", "used_at": "2025-06-10 01:06:46" }, ... ] }
    ```

---

## Image Puzzle Images

### Overview
Currently, image puzzle images are managed directly in the UI page (`pages/image_puzzle_images.php`) using direct database access. There is no dedicated backend API for this module yet. All CRUD operations and file uploads are handled within the page itself.

---

## Architectural Note

- **API-Driven Modules:**
  - Some admin pages (e.g., Promo Codes) use a backend API endpoint for all operations. This is the recommended approach for maintainability and consistency.
- **Direct DB Modules:**
  - Other pages (e.g., Image Puzzle Images) still use direct database access and local logic. These can be refactored to use an API endpoint in the future for consistency. 