<?php
// Comprehensive File Check for Postry Social Media Admin Panel
echo "🔍 Postry Social Media Admin Panel - File Check\n";
echo "===============================================\n\n";

$files = [
    // Core Firebase Files
    'firebase_config.php' => 'Firebase Configuration',
    'firebase_auth.php' => 'Firebase Authentication',
    'firebase_users.php' => 'User Management',
    'firebase_posts.php' => 'Post Management',
    'firebase_dashboard.php' => 'Dashboard Analytics',
    'firebase_notifications.php' => 'Push Notifications',
    'firebase_settings.php' => 'App Settings',
    'firebase_analytics.php' => 'Advanced Analytics',
    'firebase_simple_config.php' => 'Simplified Firebase Config',
    
    // Configuration Files
    'composer.json' => 'Composer Dependencies',
    '.htaccess' => 'Apache Configuration',
    'serviceAccountKey.json' => 'Firebase Service Account',
    'env.example' => 'Environment Variables Template',
    
    // Documentation
    'QUICK_SETUP_GUIDE.md' => 'Quick Setup Guide',
    'FIREBASE_SETUP_GUIDE.md' => 'Firebase Setup Guide',
    'FIREBASE_MIGRATION_SUMMARY.md' => 'Migration Summary',
    
    // Utility Files
    'install.php' => 'Installation Checker',
    'test_firebase.php' => 'Firebase Connection Test',
    'check_files.php' => 'File Checker (this file)'
];

$required_dirs = [
    'uploads' => 'File Upload Directory',
    'logs' => 'Log Files Directory',
    'vendor' => 'Composer Dependencies'
];

$errors = [];
$warnings = [];
$success = [];

echo "📁 Checking Files:\n";
echo "==================\n\n";

foreach ($files as $file => $description) {
    $filePath = __DIR__ . '/' . $file;
    
    if (file_exists($filePath)) {
        $size = filesize($filePath);
        $sizeFormatted = $size > 1024 ? round($size / 1024, 2) . ' KB' : $size . ' bytes';
        
        if ($size > 0) {
            $success[] = "✅ $file ($description) - $sizeFormatted";
        } else {
            $warnings[] = "⚠️  $file ($description) - Empty file";
        }
    } else {
        $errors[] = "❌ $file ($description) - Missing";
    }
}

echo "📂 Checking Directories:\n";
echo "========================\n\n";

foreach ($required_dirs as $dir => $description) {
    $dirPath = __DIR__ . '/' . $dir;
    
    if (is_dir($dirPath)) {
        if (is_writable($dirPath)) {
            $success[] = "✅ $dir/ ($description) - Exists and writable";
        } else {
            $warnings[] = "⚠️  $dir/ ($description) - Exists but not writable";
        }
    } else {
        $warnings[] = "⚠️  $dir/ ($description) - Missing";
    }
}

// Check PHP syntax
echo "🔧 Checking PHP Syntax:\n";
echo "=======================\n\n";

$phpFiles = [
    'firebase_config.php',
    'firebase_auth.php',
    'firebase_users.php',
    'firebase_posts.php',
    'firebase_dashboard.php',
    'firebase_notifications.php',
    'firebase_settings.php',
    'firebase_analytics.php',
    'firebase_simple_config.php',
    'install.php',
    'test_firebase.php'
];

foreach ($phpFiles as $phpFile) {
    $filePath = __DIR__ . '/' . $phpFile;
    
    if (file_exists($filePath)) {
        $output = [];
        $returnCode = 0;
        
        exec("php -l \"$filePath\" 2>&1", $output, $returnCode);
        
        if ($returnCode === 0) {
            $success[] = "✅ $phpFile - PHP syntax OK";
        } else {
            $errors[] = "❌ $phpFile - PHP syntax error: " . implode(' ', $output);
        }
    }
}

// Check service account key
echo "🔑 Checking Service Account Key:\n";
echo "=================================\n\n";

$serviceAccountPath = __DIR__ . '/serviceAccountKey.json';
if (file_exists($serviceAccountPath)) {
    $keyContent = file_get_contents($serviceAccountPath);
    $keyData = json_decode($keyContent, true);
    
    if ($keyData) {
        if (isset($keyData['project_id'])) {
            $success[] = "✅ Service Account Key - Valid JSON (Project: " . $keyData['project_id'] . ")";
        } else {
            $errors[] = "❌ Service Account Key - Missing project_id";
        }
        
        if (isset($keyData['private_key'])) {
            $success[] = "✅ Service Account Key - Private key present";
        } else {
            $errors[] = "❌ Service Account Key - Missing private_key";
        }
        
        if (isset($keyData['client_email'])) {
            $success[] = "✅ Service Account Key - Client email present";
        } else {
            $errors[] = "❌ Service Account Key - Missing client_email";
        }
    } else {
        $errors[] = "❌ Service Account Key - Invalid JSON";
    }
} else {
    $errors[] = "❌ Service Account Key - File not found";
}

// Check composer.json
echo "📦 Checking Composer Configuration:\n";
echo "===================================\n\n";

$composerPath = __DIR__ . '/composer.json';
if (file_exists($composerPath)) {
    $composerContent = file_get_contents($composerPath);
    $composerData = json_decode($composerContent, true);
    
    if ($composerData) {
        $success[] = "✅ composer.json - Valid JSON";
        
        if (isset($composerData['name'])) {
            $success[] = "✅ composer.json - Project name: " . $composerData['name'];
        }
        
        if (isset($composerData['require'])) {
            $success[] = "✅ composer.json - Dependencies defined";
        }
    } else {
        $errors[] = "❌ composer.json - Invalid JSON";
    }
} else {
    $errors[] = "❌ composer.json - File not found";
}

// Display results
echo "📊 File Check Results:\n";
echo "=====================\n\n";

if (!empty($success)) {
    echo "✅ SUCCESS (" . count($success) . " items):\n";
    foreach ($success as $msg) {
        echo "   $msg\n";
    }
    echo "\n";
}

if (!empty($warnings)) {
    echo "⚠️  WARNINGS (" . count($warnings) . " items):\n";
    foreach ($warnings as $msg) {
        echo "   $msg\n";
    }
    echo "\n";
}

if (!empty($errors)) {
    echo "❌ ERRORS (" . count($errors) . " items):\n";
    foreach ($errors as $msg) {
        echo "   $msg\n";
    }
    echo "\n";
}

// Final status
$totalChecks = count($success) + count($warnings) + count($errors);
$successRate = $totalChecks > 0 ? round((count($success) / $totalChecks) * 100, 1) : 0;

echo "🎯 SUMMARY:\n";
echo "===========\n";
echo "Total Checks: $totalChecks\n";
echo "Success Rate: $successRate%\n";
echo "Success: " . count($success) . "\n";
echo "Warnings: " . count($warnings) . "\n";
echo "Errors: " . count($errors) . "\n\n";

if (empty($errors)) {
    echo "🎉 FILE CHECK STATUS: ALL GOOD!\n";
    echo "===============================\n\n";
    echo "🚀 Your admin panel files are ready!\n";
    echo "📱 All Firebase backend files are properly configured.\n";
    echo "🔧 No critical issues found.\n\n";
    
    if (!empty($warnings)) {
        echo "💡 Note: Some warnings were found but they won't prevent the system from working.\n\n";
    }
    
} else {
    echo "❌ FILE CHECK STATUS: NEEDS ATTENTION\n";
    echo "=====================================\n\n";
    echo "Please fix the errors above before proceeding.\n\n";
}

echo "📋 Next Steps:\n";
echo "1. Run: php install.php (to check system requirements)\n";
echo "2. Run: php test_firebase.php (to test Firebase connection)\n";
echo "3. Create admin user and start using the admin panel\n\n";

echo "🎯 Project: Postry Social Media Admin Panel\n";
echo "🔥 Backend: Firebase Firestore\n";
echo "📱 Status: Ready for Mobile App Integration\n\n";
?>
