<?php
// Use production Firebase configuration
require_once __DIR__ . '/firebase_production.php';

header('Content-Type: application/json');
require_firebase_auth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

if ($method === 'GET' && $action === 'overview') {
    // Get overview analytics
    $userStats = FirebaseHelper::getUserStats();
    $postStats = FirebaseHelper::getPostStats();
    
    // Get engagement stats
    $totalLikes = 0;
    $totalComments = 0;
    $totalShares = 0;
    $totalViews = 0;
    
    $posts = FirebaseHelper::getCollection('posts', 1000); // Get more posts for accurate stats
    foreach ($posts as $post) {
        $totalLikes += $post['likes_count'] ?? 0;
        $totalComments += $post['comments_count'] ?? 0;
        $totalShares += $post['shares_count'] ?? 0;
        $totalViews += $post['views_count'] ?? 0;
    }
    
    // Get daily growth for last 30 days
    $dailyGrowth = [];
    for ($i = 29; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        
        $usersCreated = FirebaseHelper::getCollectionCount('users', ['created_at' => $date]);
        $postsCreated = FirebaseHelper::getCollectionCount('posts', ['created_at' => $date]);
        
        $dailyGrowth[] = [
            'date' => $date,
            'users' => $usersCreated,
            'posts' => $postsCreated
        ];
    }
    
    // Get top content
    $topPosts = FirebaseHelper::getCollection('posts', 10, ['field' => 'likes_count', 'direction' => 'desc']);
    $topUsers = FirebaseHelper::getCollection('users', 10, ['field' => 'followers_count', 'direction' => 'desc']);
    
    echo json_encode([
        'success' => true,
        'overview' => [
            'users' => $userStats,
            'posts' => $postStats,
            'engagement' => [
                'total_likes' => $totalLikes,
                'total_comments' => $totalComments,
                'total_shares' => $totalShares,
                'total_views' => $totalViews,
                'avg_engagement_rate' => $userStats['total_users'] > 0 ? round(($totalLikes + $totalComments + $totalShares) / $userStats['total_users'], 2) : 0
            ]
        ],
        'daily_growth' => $dailyGrowth,
        'top_posts' => $topPosts,
        'top_users' => $topUsers
    ]);
    exit;
}

if ($method === 'GET' && $action === 'users') {
    // Get user analytics
    $timeRange = $_GET['range'] ?? '7'; // 7, 30, 90 days
    
    $userStats = FirebaseHelper::getUserStats();
    
    // Get user growth over time
    $userGrowth = [];
    for ($i = (int)$timeRange - 1; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        $usersCreated = FirebaseHelper::getCollectionCount('users', ['created_at' => $date]);
        
        $userGrowth[] = [
            'date' => $date,
            'new_users' => $usersCreated
        ];
    }
    
    // Get user activity stats
    $activeUsers = FirebaseHelper::getCollection('users', 100, ['field' => 'last_active', 'direction' => 'desc']);
    $recentlyActive = 0;
    $oneDayAgo = date('Y-m-d H:i:s', strtotime('-1 day'));
    
    foreach ($activeUsers as $user) {
        if (isset($user['last_active']) && $user['last_active'] >= $oneDayAgo) {
            $recentlyActive++;
        }
    }
    
    // Get user demographics (if available)
    $verifiedUsers = FirebaseHelper::getCollectionCount('users', ['is_verified' => true]);
    $blockedUsers = FirebaseHelper::getCollectionCount('users', ['is_blocked' => true]);
    
    echo json_encode([
        'success' => true,
        'user_analytics' => [
            'total_users' => $userStats['total_users'],
            'active_users' => $userStats['active_users'],
            'blocked_users' => $userStats['blocked_users'],
            'verified_users' => $verifiedUsers,
            'recently_active' => $recentlyActive,
            'growth_rate' => $timeRange > 1 ? round($userGrowth[count($userGrowth) - 1]['new_users'] / max($userGrowth[0]['new_users'], 1) * 100, 2) : 0
        ],
        'user_growth' => $userGrowth
    ]);
    exit;
}

if ($method === 'GET' && $action === 'posts') {
    // Get post analytics
    $timeRange = $_GET['range'] ?? '7'; // 7, 30, 90 days
    
    $postStats = FirebaseHelper::getPostStats();
    
    // Get post growth over time
    $postGrowth = [];
    for ($i = (int)$timeRange - 1; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        $postsCreated = FirebaseHelper::getCollectionCount('posts', ['created_at' => $date]);
        
        $postGrowth[] = [
            'date' => $date,
            'new_posts' => $postsCreated
        ];
    }
    
    // Get engagement metrics
    $posts = FirebaseHelper::getCollection('posts', 1000);
    $engagementMetrics = [
        'total_likes' => 0,
        'total_comments' => 0,
        'total_shares' => 0,
        'total_views' => 0,
        'avg_likes_per_post' => 0,
        'avg_comments_per_post' => 0,
        'avg_shares_per_post' => 0,
        'avg_views_per_post' => 0
    ];
    
    foreach ($posts as $post) {
        $engagementMetrics['total_likes'] += $post['likes_count'] ?? 0;
        $engagementMetrics['total_comments'] += $post['comments_count'] ?? 0;
        $engagementMetrics['total_shares'] += $post['shares_count'] ?? 0;
        $engagementMetrics['total_views'] += $post['views_count'] ?? 0;
    }
    
    $postCount = count($posts);
    if ($postCount > 0) {
        $engagementMetrics['avg_likes_per_post'] = round($engagementMetrics['total_likes'] / $postCount, 2);
        $engagementMetrics['avg_comments_per_post'] = round($engagementMetrics['total_comments'] / $postCount, 2);
        $engagementMetrics['avg_shares_per_post'] = round($engagementMetrics['total_shares'] / $postCount, 2);
        $engagementMetrics['avg_views_per_post'] = round($engagementMetrics['total_views'] / $postCount, 2);
    }
    
    // Get content types distribution
    $contentTypes = [];
    foreach ($posts as $post) {
        $type = $post['media_type'] ?? 'text';
        $contentTypes[$type] = ($contentTypes[$type] ?? 0) + 1;
    }
    
    echo json_encode([
        'success' => true,
        'post_analytics' => [
            'total_posts' => $postStats['total_posts'],
            'active_posts' => $postStats['active_posts'],
            'reported_posts' => $postStats['reported_posts'],
            'engagement_metrics' => $engagementMetrics,
            'content_types' => $contentTypes
        ],
        'post_growth' => $postGrowth
    ]);
    exit;
}

if ($method === 'GET' && $action === 'engagement') {
    // Get engagement analytics
    $posts = FirebaseHelper::getCollection('posts', 1000);
    
    $engagementData = [];
    foreach ($posts as $post) {
        $engagementData[] = [
            'post_id' => $post['id'],
            'likes' => $post['likes_count'] ?? 0,
            'comments' => $post['comments_count'] ?? 0,
            'shares' => $post['shares_count'] ?? 0,
            'views' => $post['views_count'] ?? 0,
            'created_at' => $post['created_at'] ?? ''
        ];
    }
    
    // Sort by total engagement
    usort($engagementData, function($a, $b) {
        $aTotal = $a['likes'] + $a['comments'] + $a['shares'];
        $bTotal = $b['likes'] + $b['comments'] + $b['shares'];
        return $bTotal - $aTotal;
    });
    
    // Get top 10 most engaging posts
    $topEngagingPosts = array_slice($engagementData, 0, 10);
    
    // Calculate engagement trends
    $engagementTrends = [];
    for ($i = 6; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        $dailyEngagement = 0;
        
        foreach ($posts as $post) {
            if (isset($post['created_at']) && date('Y-m-d', strtotime($post['created_at'])) === $date) {
                $dailyEngagement += ($post['likes_count'] ?? 0) + ($post['comments_count'] ?? 0) + ($post['shares_count'] ?? 0);
            }
        }
        
        $engagementTrends[] = [
            'date' => $date,
            'engagement' => $dailyEngagement
        ];
    }
    
    echo json_encode([
        'success' => true,
        'engagement_analytics' => [
            'top_engaging_posts' => $topEngagingPosts,
            'engagement_trends' => $engagementTrends,
            'total_engagement' => array_sum(array_column($engagementData, 'likes')) + 
                                array_sum(array_column($engagementData, 'comments')) + 
                                array_sum(array_column($engagementData, 'shares'))
        ]
    ]);
    exit;
}

if ($method === 'GET' && $action === 'reports') {
    // Get reports and moderation analytics
    $reportedPosts = FirebaseHelper::getCollection('posts', 100, ['field' => 'updated_at', 'direction' => 'desc'], ['is_reported' => true]);
    $blockedUsers = FirebaseHelper::getCollection('users', 100, ['field' => 'updated_at', 'direction' => 'desc'], ['is_blocked' => true]);
    
    // Get report reasons distribution
    $reportReasons = [];
    foreach ($reportedPosts as $post) {
        $reason = $post['report_reason'] ?? 'Unknown';
        $reportReasons[$reason] = ($reportReasons[$reason] ?? 0) + 1;
    }
    
    // Get moderation actions over time
    $moderationActions = [];
    for ($i = 29; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-{$i} days"));
        
        $postsReported = 0;
        $usersBlocked = 0;
        
        foreach ($reportedPosts as $post) {
            if (isset($post['updated_at']) && date('Y-m-d', strtotime($post['updated_at'])) === $date) {
                $postsReported++;
            }
        }
        
        foreach ($blockedUsers as $user) {
            if (isset($user['updated_at']) && date('Y-m-d', strtotime($user['updated_at'])) === $date) {
                $usersBlocked++;
            }
        }
        
        $moderationActions[] = [
            'date' => $date,
            'posts_reported' => $postsReported,
            'users_blocked' => $usersBlocked
        ];
    }
    
    echo json_encode([
        'success' => true,
        'reports_analytics' => [
            'total_reported_posts' => count($reportedPosts),
            'total_blocked_users' => count($blockedUsers),
            'report_reasons' => $reportReasons,
            'moderation_actions' => $moderationActions
        ]
    ]);
    exit;
}

if ($method === 'GET' && $action === 'export') {
    // Export analytics data
    $format = $_GET['format'] ?? 'json'; // json, csv
    $type = $_GET['type'] ?? 'overview'; // overview, users, posts, engagement
    
    $data = [];
    
    switch ($type) {
        case 'users':
            $data = FirebaseHelper::getUsers(1000);
            break;
        case 'posts':
            $data = FirebaseHelper::getPosts(1000);
            break;
        case 'engagement':
            $posts = FirebaseHelper::getCollection('posts', 1000);
            foreach ($posts as $post) {
                $data[] = [
                    'post_id' => $post['id'],
                    'likes' => $post['likes_count'] ?? 0,
                    'comments' => $post['comments_count'] ?? 0,
                    'shares' => $post['shares_count'] ?? 0,
                    'views' => $post['views_count'] ?? 0,
                    'created_at' => $post['created_at'] ?? ''
                ];
            }
            break;
        default:
            $data = [
                'users' => FirebaseHelper::getUserStats(),
                'posts' => FirebaseHelper::getPostStats(),
                'exported_at' => date('Y-m-d H:i:s')
            ];
            break;
    }
    
    if ($format === 'csv' && is_array($data) && !empty($data)) {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="analytics_' . $type . '_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        if (!empty($data)) {
            // Write headers
            fputcsv($output, array_keys($data[0]));
            
            // Write data
            foreach ($data as $row) {
                fputcsv($output, $row);
            }
        }
        
        fclose($output);
    } else {
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="analytics_' . $type . '_' . date('Y-m-d') . '.json"');
        echo json_encode($data, JSON_PRETTY_PRINT);
    }
    exit;
}

http_response_code(405);
echo json_encode(['error' => 'Method not allowed']);
?>
