<?php
// Use production Firebase configuration
require_once __DIR__ . '/firebase_production.php';

header('Content-Type: application/json');

function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

function getAdminByEmail($email) {
    try {
        $auth = FirebaseConfig::getAuth();
        $user = $auth->getUserByEmail($email);
        return $user;
    } catch (Exception $e) {
        error_log('Error getting admin by email: ' . $e->getMessage());
        return null;
    }
}

function createAdminUser($email, $password) {
    try {
        $auth = FirebaseConfig::getAuth();
        $userProperties = [
            'email' => $email,
            'password' => $password,
            'emailVerified' => true,
            'disabled' => false
        ];
        
        $user = $auth->createUser($userProperties);
        
        // Set admin custom claims
        $auth->setCustomUserClaims($user->uid, ['admin' => true]);
        
        return $user;
    } catch (Exception $e) {
        error_log('Error creating admin user: ' . $e->getMessage());
        return null;
    }
}

function verifyAdminToken($token) {
    try {
        $auth = FirebaseConfig::getAuth();
        $verifiedIdToken = $auth->verifyIdToken($token);
        $uid = $verifiedIdToken->claims()->get('sub');
        
        // Check if user is admin
        $user = $auth->getUser($uid);
        $customClaims = $user->customClaims;
        
        return isset($customClaims['admin']) && $customClaims['admin'] === true;
    } catch (Exception $e) {
        error_log('Error verifying admin token: ' . $e->getMessage());
        return false;
    }
}

function signInWithEmailAndPassword($email, $password) {
    try {
        // This would typically be done on the client side
        // For admin panel, we'll use Firebase Admin SDK to verify credentials
        $auth = FirebaseConfig::getAuth();
        $user = $auth->getUserByEmail($email);
        
        if ($user && !$user->disabled) {
            // Check if user is admin
            $customClaims = $user->customClaims;
            if (isset($customClaims['admin']) && $customClaims['admin'] === true) {
                // Generate a custom token for admin
                $customToken = $auth->createCustomToken($user->uid, ['admin' => true]);
                return [
                    'success' => true,
                    'uid' => $user->uid,
                    'email' => $user->email,
                    'custom_token' => $customToken->toString()
                ];
            }
        }
        
        return ['success' => false, 'error' => 'Invalid credentials or not admin'];
    } catch (Exception $e) {
        error_log('Error signing in: ' . $e->getMessage());
        return ['success' => false, 'error' => 'Authentication failed'];
    }
}

$method = $_SERVER['REQUEST_METHOD'];
$path = $_GET['action'] ?? '';

if ($method === 'POST' && $path === 'login') {
    $data = json_decode(file_get_contents('php://input'), true);
    $email = $data['email'] ?? '';
    $password = $data['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode(['error' => 'Email and password required']);
        exit;
    }
    
    $result = signInWithEmailAndPassword($email, $password);
    
    if ($result['success']) {
        echo json_encode([
            'success' => true,
            'uid' => $result['uid'],
            'email' => $result['email'],
            'custom_token' => $result['custom_token']
        ]);
    } else {
        http_response_code(401);
        echo json_encode(['error' => $result['error']]);
    }
    exit;
}

if ($method === 'POST' && $path === 'logout') {
    // For Firebase, logout is typically handled on client side
    echo json_encode(['success' => true, 'message' => 'Logged out successfully']);
    exit;
}

if ($method === 'POST' && $path === 'create_admin') {
    $data = json_decode(file_get_contents('php://input'), true);
    $email = $data['email'] ?? '';
    $password = $data['password'] ?? '';
    
    if (empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode(['error' => 'Email and password required']);
        exit;
    }
    
    $user = createAdminUser($email, $password);
    
    if ($user) {
        echo json_encode([
            'success' => true,
            'uid' => $user->uid,
            'email' => $user->email,
            'message' => 'Admin user created successfully'
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to create admin user']);
    }
    exit;
}

// Token and session validation for other endpoints
function require_firebase_auth() {
    $headers = getallheaders();
    $token = $headers['Authorization'] ?? '';
    
    // Remove 'Bearer ' prefix if present
    $token = str_replace('Bearer ', '', $token);
    
    if (empty($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Authorization token required']);
        exit;
    }
    
    $isValid = verifyAdminToken($token);
    
    if (!$isValid) {
        http_response_code(401);
        echo json_encode(['error' => 'Invalid or expired token']);
        exit;
    }
    
    return true;
}

// Fallback for invalid requests
http_response_code(405);
echo json_encode(['error' => 'Method not allowed']);
?>
