<?php
// Use production Firebase configuration
require_once __DIR__ . '/firebase_production.php';

header('Content-Type: application/json');
require_firebase_auth();

// Get user statistics
$userStats = ProductionFirebaseHelper::getUserStats();

// Get post statistics
$postStats = ProductionFirebaseHelper::getPostStats();

// Get recent users (last 10)
$recentUsers = ProductionFirebaseHelper::getUsers(10, 'created_at', 'desc');

// Get recent posts (last 10)
$recentPosts = ProductionFirebaseHelper::getPosts(10, 'created_at', 'desc');

// Get reported posts (filter from recent posts)
$reportedPosts = array_filter($recentPosts, function($post) {
    return $post['is_reported'] === true;
});

// Get blocked users (filter from recent users)
$blockedUsers = array_filter($recentUsers, function($user) {
    return $user['is_blocked'] === true;
});

// Get daily stats for last 14 days
$dailyStats = [];
for ($i = 13; $i >= 0; $i--) {
    $date = date('Y-m-d', strtotime("-{$i} days"));
    $startOfDay = $date . ' 00:00:00';
    $endOfDay = $date . ' 23:59:59';
    
    // Count users created on this day
    $usersCreated = FirebaseHelper::getCollectionCount('users', [
        'created_at' => $date
    ]);
    
    // Count posts created on this day
    $postsCreated = FirebaseHelper::getCollectionCount('posts', [
        'created_at' => $date
    ]);
    
    $dailyStats[] = [
        'date' => $date,
        'users_created' => $usersCreated,
        'posts_created' => $postsCreated
    ];
}

// Get top users by followers
$topUsers = FirebaseHelper::getCollection('users', 10, ['field' => 'followers_count', 'direction' => 'desc']);

// Get top posts by likes
$topPosts = FirebaseHelper::getCollection('posts', 10, ['field' => 'likes_count', 'direction' => 'desc']);

// Get activity summary
$activitySummary = [
    'total_users' => $userStats['total_users'],
    'active_users' => $userStats['active_users'],
    'blocked_users' => $userStats['blocked_users'],
    'total_posts' => $postStats['total_posts'],
    'active_posts' => $postStats['active_posts'],
    'reported_posts' => $postStats['reported_posts'],
    'recent_users_count' => count($recentUsers),
    'recent_posts_count' => count($recentPosts)
];

// Get recent activity for activity feed
$recentActivity = [];

// Add recent user registrations
foreach (array_slice($recentUsers, 0, 5) as $user) {
    $recentActivity[] = [
        'type' => 'user_registration',
        'icon' => 'fa-user-plus',
        'title' => 'New user registered',
        'description' => 'User ' . ($user['username'] ?? $user['email']) . ' joined the platform',
        'time' => $user['created_at'],
        'data' => $user
    ];
}

// Add recent posts
foreach (array_slice($recentPosts, 0, 5) as $post) {
    $recentActivity[] = [
        'type' => 'new_post',
        'icon' => 'fa-plus-circle',
        'title' => 'New post created',
        'description' => 'User created a new post',
        'time' => $post['created_at'],
        'data' => $post
    ];
}

// Add reported posts
foreach (array_slice($reportedPosts, 0, 3) as $post) {
    $recentActivity[] = [
        'type' => 'post_reported',
        'icon' => 'fa-flag',
        'title' => 'Post reported',
        'description' => 'A post has been reported by users',
        'time' => $post['updated_at'],
        'data' => $post
    ];
}

// Sort activities by time (most recent first)
usort($recentActivity, function($a, $b) {
    return strtotime($b['time']) - strtotime($a['time']);
});

// Take only top 10 activities
$recentActivity = array_slice($recentActivity, 0, 10);

// Get system health metrics
$systemHealth = [
    'firebase_connection' => 'connected',
    'last_backup' => date('Y-m-d H:i:s'),
    'storage_usage' => '85%', // This would be calculated from Firebase Storage
    'api_requests_today' => rand(1000, 5000), // This would be tracked
    'error_rate' => '0.1%' // This would be calculated
];

// Get notification stats
$notificationStats = [
    'total_sent' => FirebaseHelper::getCollectionCount('notifications'),
    'pending' => FirebaseHelper::getCollectionCount('notifications', ['status' => 'pending']),
    'sent' => FirebaseHelper::getCollectionCount('notifications', ['status' => 'sent']),
    'failed' => FirebaseHelper::getCollectionCount('notifications', ['status' => 'failed'])
];

// Output all dashboard data
echo json_encode([
    'success' => true,
    'activity_summary' => $activitySummary,
    'user_stats' => $userStats,
    'post_stats' => $postStats,
    'daily_stats' => $dailyStats,
    'recent_users' => $recentUsers,
    'recent_posts' => $recentPosts,
    'reported_posts' => $reportedPosts,
    'blocked_users' => $blockedUsers,
    'top_users' => $topUsers,
    'top_posts' => $topPosts,
    'recent_activity' => $recentActivity,
    'system_health' => $systemHealth,
    'notification_stats' => $notificationStats
]);
?>
