<?php
session_start();
require_once __DIR__ . '/../backend/firebase_config.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login_firebase.php');
    exit;
}

// Handle actions
$message = '';
$messageType = '';

if ($_POST) {
    $action = $_POST['action'] ?? '';
    $postId = $_POST['post_id'] ?? '';
    
    try {
        switch ($action) {
            case 'hide_post':
                FirebaseHelper::updateDocument('posts', $postId, [
                    'isActive' => false,
                    'hiddenAt' => new DateTime(),
                    'hiddenBy' => $_SESSION['admin_id']
                ]);
                $message = 'Post hidden successfully';
                $messageType = 'success';
                break;
                
            case 'show_post':
                FirebaseHelper::updateDocument('posts', $postId, [
                    'isActive' => true,
                    'shownAt' => new DateTime(),
                    'shownBy' => $_SESSION['admin_id']
                ]);
                $message = 'Post shown successfully';
                $messageType = 'success';
                break;
                
            case 'delete_post':
                FirebaseHelper::deleteDocument('posts', $postId);
                $message = 'Post deleted successfully';
                $messageType = 'success';
                break;
        }
    } catch (Exception $e) {
        $message = 'Error: ' . $e->getMessage();
        $messageType = 'error';
    }
}

// Get posts with pagination and search
$page = $_GET['page'] ?? 1;
$limit = 15;
$search = $_GET['search'] ?? '';
$filter = $_GET['filter'] ?? 'all';

try {
    if ($search) {
        $posts = FirebaseHelper::searchDocuments('posts', 'postText', $search, $limit);
    } else {
        $where = [];
        if ($filter === 'active') {
            $where['isActive'] = true;
        } elseif ($filter === 'hidden') {
            $where['isActive'] = false;
        } elseif ($filter === 'reported') {
            $where['isReported'] = true;
        }
        
        $posts = FirebaseHelper::getCollection('posts', $limit, ['field' => 'createdOn', 'direction' => 'DESC'], $where);
    }
    
    $totalPosts = FirebaseHelper::getCollectionCount('posts');
    $activePosts = FirebaseHelper::getCollectionCount('posts', ['isActive' => true]);
    $hiddenPosts = FirebaseHelper::getCollectionCount('posts', ['isActive' => false]);
    $reportedPosts = FirebaseHelper::getCollectionCount('posts', ['isReported' => true]);
    
} catch (Exception $e) {
    $posts = [];
    $totalPosts = $activePosts = $hiddenPosts = $reportedPosts = 0;
    $message = 'Error loading posts: ' . $e->getMessage();
    $messageType = 'error';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Posts Management - Postery Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
    :root {
        --primary: #6366f1;
        --primary-dark: #4f46e5;
        --secondary: #8b5cf6;
        --success: #10b981;
        --danger: #ef4444;
        --warning: #f59e0b;
        --info: #3b82f6;
        --light: #f8fafc;
        --dark: #1e293b;
        --gray: #64748b;
        --gray-light: #e2e8f0;
        --sidebar-width: 280px;
        --card-radius: 12px;
        --transition: all 0.3s ease;
    }

    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
        font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: #1e293b;
        overflow-x: hidden;
        line-height: 1.6;
        min-height: 100vh;
    }

    .admin-container {
        display: flex;
        min-height: 100vh;
    }

    .main-content {
        flex: 1;
        padding: 2rem;
        margin-left: var(--sidebar-width);
        transition: var(--transition);
    }

    .page-header {
        background: white;
        border-radius: var(--card-radius);
        padding: 2rem;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .page-title {
        font-size: 2rem;
        font-weight: 700;
        color: var(--dark);
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
    }

    .page-title i {
        margin-right: 1rem;
        color: var(--primary);
    }

    .page-subtitle {
        color: var(--gray);
        font-size: 1rem;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        border-radius: var(--card-radius);
        padding: 1.5rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        text-align: center;
        transition: var(--transition);
    }

    .stat-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
    }

    .stat-number {
        font-size: 2rem;
        font-weight: 800;
        color: var(--primary);
        margin-bottom: 0.5rem;
    }

    .stat-label {
        color: var(--gray);
        font-size: 0.9rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .filters-section {
        background: white;
        border-radius: var(--card-radius);
        padding: 1.5rem;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .filters-row {
        display: flex;
        gap: 1rem;
        align-items: center;
        flex-wrap: wrap;
    }

    .search-box {
        flex: 1;
        min-width: 300px;
        position: relative;
    }

    .search-input {
        width: 100%;
        padding: 1rem 1rem 1rem 3rem;
        border: 2px solid var(--gray-light);
        border-radius: 12px;
        font-size: 1rem;
        transition: var(--transition);
        background: #f9fafb;
    }

    .search-input:focus {
        outline: none;
        border-color: var(--primary);
        background: white;
        box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
    }

    .search-icon {
        position: absolute;
        left: 1rem;
        top: 50%;
        transform: translateY(-50%);
        color: var(--gray);
        font-size: 1.1rem;
    }

    .filter-select {
        padding: 1rem;
        border: 2px solid var(--gray-light);
        border-radius: 12px;
        font-size: 1rem;
        background: white;
        cursor: pointer;
        transition: var(--transition);
    }

    .filter-select:focus {
        outline: none;
        border-color: var(--primary);
    }

    .posts-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .post-card {
        background: white;
        border-radius: var(--card-radius);
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        overflow: hidden;
        transition: var(--transition);
    }

    .post-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
    }

    .post-image {
        width: 100%;
        height: 200px;
        object-fit: cover;
        background: var(--gray-light);
    }

    .post-content {
        padding: 1.5rem;
    }

    .post-header {
        display: flex;
        align-items: center;
        margin-bottom: 1rem;
    }

    .post-avatar {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        object-fit: cover;
        margin-right: 1rem;
    }

    .post-user-info h4 {
        font-weight: 600;
        color: var(--dark);
        margin-bottom: 0.2rem;
    }

    .post-user-info p {
        color: var(--gray);
        font-size: 0.8rem;
    }

    .post-text {
        color: var(--dark);
        margin-bottom: 1rem;
        line-height: 1.5;
    }

    .post-stats {
        display: flex;
        gap: 1rem;
        margin-bottom: 1rem;
        font-size: 0.9rem;
        color: var(--gray);
    }

    .post-actions {
        display: flex;
        gap: 0.5rem;
        flex-wrap: wrap;
    }

    .btn {
        padding: 0.5rem 1rem;
        border: none;
        border-radius: 8px;
        font-size: 0.8rem;
        font-weight: 600;
        cursor: pointer;
        transition: var(--transition);
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.3rem;
    }

    .btn-primary { background: var(--primary); color: white; }
    .btn-success { background: var(--success); color: white; }
    .btn-danger { background: var(--danger); color: white; }
    .btn-warning { background: var(--warning); color: white; }
    .btn-info { background: var(--info); color: white; }

    .btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
    }

    .status-badge {
        display: inline-block;
        padding: 0.3rem 0.8rem;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-active { background: #dcfce7; color: #166534; }
    .status-hidden { background: #fef2f2; color: #dc2626; }
    .status-reported { background: #fef3c7; color: #d97706; }

    .alert {
        padding: 1rem 1.5rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .alert-success {
        background: #dcfce7;
        color: #166534;
        border: 1px solid #bbf7d0;
    }

    .alert-error {
        background: #fef2f2;
        color: #dc2626;
        border: 1px solid #fecaca;
    }

    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        color: var(--gray);
        background: white;
        border-radius: var(--card-radius);
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .empty-state i {
        font-size: 4rem;
        margin-bottom: 1rem;
        color: var(--gray-light);
    }

    .empty-state h3 {
        font-size: 1.5rem;
        margin-bottom: 0.5rem;
        color: var(--dark);
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .main-content {
            padding: 1.5rem;
        }
    }

    @media (max-width: 900px) {
        .main-content {
            margin-left: 0;
            padding: 1rem;
        }
        .admin-container {
            flex-direction: column;
        }
        .filters-row {
            flex-direction: column;
            align-items: stretch;
        }
        .search-box {
            min-width: auto;
        }
        .posts-grid {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 600px) {
        .main-content {
            padding: 0.75rem;
        }
        .page-header {
            padding: 1.5rem;
        }
        .page-title {
            font-size: 1.5rem;
        }
        .stats-grid {
            grid-template-columns: 1fr 1fr;
        }
        .post-actions {
            flex-direction: column;
        }
    }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include '../sidebar_new.php'; ?>
        <main class="main-content">
            
            <!-- Page Header -->
            <div class="page-header">
                <h1 class="page-title">
                    <i class="fas fa-images"></i>
                    Posts Management
                </h1>
                <p class="page-subtitle">Manage and moderate user posts on your social media platform</p>
            </div>

            <!-- Alert Messages -->
            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?>">
                    <i class="fas fa-<?= $messageType === 'success' ? 'check-circle' : 'exclamation-triangle' ?>"></i>
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php endif; ?>

            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($totalPosts) ?></div>
                    <div class="stat-label">Total Posts</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($activePosts) ?></div>
                    <div class="stat-label">Active Posts</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($hiddenPosts) ?></div>
                    <div class="stat-label">Hidden Posts</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($reportedPosts) ?></div>
                    <div class="stat-label">Reported Posts</div>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="filters-section">
                <form method="GET" class="filters-row">
                    <div class="search-box">
                        <i class="fas fa-search search-icon"></i>
                        <input 
                            type="text" 
                            name="search" 
                            class="search-input" 
                            placeholder="Search posts by content..."
                            value="<?= htmlspecialchars($search) ?>"
                        >
                    </div>
                    <select name="filter" class="filter-select">
                        <option value="all" <?= $filter === 'all' ? 'selected' : '' ?>>All Posts</option>
                        <option value="active" <?= $filter === 'active' ? 'selected' : '' ?>>Active Only</option>
                        <option value="hidden" <?= $filter === 'hidden' ? 'selected' : '' ?>>Hidden Only</option>
                        <option value="reported" <?= $filter === 'reported' ? 'selected' : '' ?>>Reported Only</option>
                    </select>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter"></i>
                        Filter
                    </button>
                </form>
            </div>

            <!-- Posts Grid -->
            <?php if (empty($posts)): ?>
                <div class="empty-state">
                    <i class="fas fa-images"></i>
                    <h3>No posts found</h3>
                    <p>Try adjusting your search criteria or filters</p>
                </div>
            <?php else: ?>
                <div class="posts-grid">
                    <?php foreach ($posts as $post): ?>
                        <div class="post-card">
                            <?php if ($post['postImage'] ?? false): ?>
                                <img 
                                    src="<?= htmlspecialchars($post['postImage']) ?>" 
                                    alt="Post Image" 
                                    class="post-image"
                                    onerror="this.style.display='none'"
                                >
                            <?php endif; ?>
                            
                            <div class="post-content">
                                <div class="post-header">
                                    <img 
                                        src="<?= htmlspecialchars($post['postUserImage'] ?? 'https://ui-avatars.com/api/?name=' . urlencode($post['postUserName'] ?? 'User') . '&background=6366f1&color=fff&rounded=true&size=40') ?>" 
                                        alt="User Avatar" 
                                        class="post-avatar"
                                    >
                                    <div class="post-user-info">
                                        <h4><?= htmlspecialchars($post['postUserName'] ?? 'Unknown User') ?></h4>
                                        <p>
                                            <?php
                                            if (isset($post['createdOn'])) {
                                                $timestamp = $post['createdOn']->getTimestamp();
                                                echo date('M j, Y g:i A', $timestamp);
                                            } else {
                                                echo 'Recently';
                                            }
                                            ?>
                                        </p>
                                    </div>
                                </div>
                                
                                <div class="post-text">
                                    <?= htmlspecialchars(substr($post['postText'] ?? '', 0, 150)) ?>
                                    <?php if (strlen($post['postText'] ?? '') > 150): ?>...<?php endif; ?>
                                </div>
                                
                                <div class="post-stats">
                                    <span><i class="fas fa-heart"></i> <?= number_format($post['usersLikes'] ?? 0) ?></span>
                                    <span><i class="fas fa-comment"></i> <?= number_format($post['commentsCount'] ?? 0) ?></span>
                                    <span><i class="fas fa-share"></i> <?= number_format($post['usersShares'] ?? 0) ?></span>
                                </div>
                                
                                <div class="post-actions">
                                    <?php
                                    $statuses = [];
                                    if ($post['isActive'] ?? true) {
                                        $statuses[] = '<span class="status-badge status-active">Active</span>';
                                    } else {
                                        $statuses[] = '<span class="status-badge status-hidden">Hidden</span>';
                                    }
                                    if ($post['isReported'] ?? false) {
                                        $statuses[] = '<span class="status-badge status-reported">Reported</span>';
                                    }
                                    echo implode(' ', $statuses);
                                    ?>
                                    
                                    <a href="post_details_firebase.php?id=<?= $post['id'] ?>" class="btn btn-info">
                                        <i class="fas fa-eye"></i>
                                        View
                                    </a>
                                    
                                    <?php if ($post['isActive'] ?? true): ?>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="hide_post">
                                            <input type="hidden" name="post_id" value="<?= $post['id'] ?>">
                                            <button type="submit" class="btn btn-warning" onclick="return confirm('Hide this post?')">
                                                <i class="fas fa-eye-slash"></i>
                                                Hide
                                            </button>
                                        </form>
                                    <?php else: ?>
                                        <form method="POST" style="display: inline;">
                                            <input type="hidden" name="action" value="show_post">
                                            <input type="hidden" name="post_id" value="<?= $post['id'] ?>">
                                            <button type="submit" class="btn btn-success" onclick="return confirm('Show this post?')">
                                                <i class="fas fa-eye"></i>
                                                Show
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                    
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="action" value="delete_post">
                                        <input type="hidden" name="post_id" value="<?= $post['id'] ?>">
                                        <button type="submit" class="btn btn-danger" onclick="return confirm('Delete this post permanently? This action cannot be undone.')">
                                            <i class="fas fa-trash"></i>
                                            Delete
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </main>
    </div>
    
    <script>
    // Auto-submit form on filter change
    document.querySelector('.filter-select').addEventListener('change', function() {
        this.form.submit();
    });
    
    // Add loading state to buttons
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function() {
            const button = this.querySelector('button[type="submit"]');
            if (button) {
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                button.disabled = true;
            }
        });
    });
    </script>
</body>
</html>
