<?php
session_start();
require_once __DIR__ . '/../backend/firebase_config.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login_firebase.php');
    exit;
}

// Handle actions
$message = '';
$messageType = '';

if ($_POST) {
    $action = $_POST['action'] ?? '';
    $userId = $_POST['user_id'] ?? '';
    
    try {
        switch ($action) {
            case 'block_user':
                FirebaseHelper::updateDocument('users', $userId, [
                    'isBlocked' => true,
                    'blockedAt' => new DateTime(),
                    'blockedBy' => $_SESSION['admin_id']
                ]);
                $message = 'User blocked successfully';
                $messageType = 'success';
                break;
                
            case 'unblock_user':
                FirebaseHelper::updateDocument('users', $userId, [
                    'isBlocked' => false,
                    'unblockedAt' => new DateTime(),
                    'unblockedBy' => $_SESSION['admin_id']
                ]);
                $message = 'User unblocked successfully';
                $messageType = 'success';
                break;
                
            case 'verify_user':
                FirebaseHelper::updateDocument('users', $userId, [
                    'isVerified' => true,
                    'verifiedAt' => new DateTime(),
                    'verifiedBy' => $_SESSION['admin_id']
                ]);
                $message = 'User verified successfully';
                $messageType = 'success';
                break;
                
            case 'delete_user':
                FirebaseHelper::deleteDocument('users', $userId);
                $message = 'User deleted successfully';
                $messageType = 'success';
                break;
        }
    } catch (Exception $e) {
        $message = 'Error: ' . $e->getMessage();
        $messageType = 'error';
    }
}

// Get users with pagination and search
$page = $_GET['page'] ?? 1;
$limit = 20;
$search = $_GET['search'] ?? '';
$filter = $_GET['filter'] ?? 'all';

$offset = ($page - 1) * $limit;

try {
    if ($search) {
        $users = FirebaseHelper::searchDocuments('users', 'display_name', $search, $limit);
    } else {
        $where = [];
        if ($filter === 'blocked') {
            $where['isBlocked'] = true;
        } elseif ($filter === 'verified') {
            $where['isVerified'] = true;
        } elseif ($filter === 'active') {
            $where['isActive'] = true;
        }
        
        $users = FirebaseHelper::getCollection('users', $limit, ['field' => 'created_time', 'direction' => 'DESC'], $where);
    }
    
    $totalUsers = FirebaseHelper::getCollectionCount('users');
    $blockedUsers = FirebaseHelper::getCollectionCount('users', ['isBlocked' => true]);
    $verifiedUsers = FirebaseHelper::getCollectionCount('users', ['isVerified' => true]);
    $activeUsers = FirebaseHelper::getCollectionCount('users', ['isActive' => true]);
    
} catch (Exception $e) {
    $users = [];
    $totalUsers = $blockedUsers = $verifiedUsers = $activeUsers = 0;
    $message = 'Error loading users: ' . $e->getMessage();
    $messageType = 'error';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Users Management - Postery Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
    :root {
        --primary: #6366f1;
        --primary-dark: #4f46e5;
        --secondary: #8b5cf6;
        --success: #10b981;
        --danger: #ef4444;
        --warning: #f59e0b;
        --info: #3b82f6;
        --light: #f8fafc;
        --dark: #1e293b;
        --gray: #64748b;
        --gray-light: #e2e8f0;
        --sidebar-width: 280px;
        --card-radius: 12px;
        --transition: all 0.3s ease;
    }

    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
        font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: #1e293b;
        overflow-x: hidden;
        line-height: 1.6;
        min-height: 100vh;
    }

    .admin-container {
        display: flex;
        min-height: 100vh;
    }

    .main-content {
        flex: 1;
        padding: 2rem;
        margin-left: var(--sidebar-width);
        transition: var(--transition);
    }

    .page-header {
        background: white;
        border-radius: var(--card-radius);
        padding: 2rem;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .page-title {
        font-size: 2rem;
        font-weight: 700;
        color: var(--dark);
        margin-bottom: 0.5rem;
        display: flex;
        align-items: center;
    }

    .page-title i {
        margin-right: 1rem;
        color: var(--primary);
    }

    .page-subtitle {
        color: var(--gray);
        font-size: 1rem;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        border-radius: var(--card-radius);
        padding: 1.5rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        text-align: center;
        transition: var(--transition);
    }

    .stat-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
    }

    .stat-number {
        font-size: 2rem;
        font-weight: 800;
        color: var(--primary);
        margin-bottom: 0.5rem;
    }

    .stat-label {
        color: var(--gray);
        font-size: 0.9rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .filters-section {
        background: white;
        border-radius: var(--card-radius);
        padding: 1.5rem;
        margin-bottom: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .filters-row {
        display: flex;
        gap: 1rem;
        align-items: center;
        flex-wrap: wrap;
    }

    .search-box {
        flex: 1;
        min-width: 300px;
        position: relative;
    }

    .search-input {
        width: 100%;
        padding: 1rem 1rem 1rem 3rem;
        border: 2px solid var(--gray-light);
        border-radius: 12px;
        font-size: 1rem;
        transition: var(--transition);
        background: #f9fafb;
    }

    .search-input:focus {
        outline: none;
        border-color: var(--primary);
        background: white;
        box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
    }

    .search-icon {
        position: absolute;
        left: 1rem;
        top: 50%;
        transform: translateY(-50%);
        color: var(--gray);
        font-size: 1.1rem;
    }

    .filter-select {
        padding: 1rem;
        border: 2px solid var(--gray-light);
        border-radius: 12px;
        font-size: 1rem;
        background: white;
        cursor: pointer;
        transition: var(--transition);
    }

    .filter-select:focus {
        outline: none;
        border-color: var(--primary);
    }

    .users-table-container {
        background: white;
        border-radius: var(--card-radius);
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        overflow: hidden;
    }

    .table-header {
        padding: 1.5rem;
        border-bottom: 1px solid var(--gray-light);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .table-title {
        font-size: 1.2rem;
        font-weight: 700;
        color: var(--dark);
    }

    .users-table {
        width: 100%;
        border-collapse: collapse;
    }

    .users-table th {
        background: #f8fafc;
        padding: 1rem;
        text-align: left;
        font-weight: 600;
        color: var(--dark);
        border-bottom: 1px solid var(--gray-light);
    }

    .users-table td {
        padding: 1rem;
        border-bottom: 1px solid var(--gray-light);
        vertical-align: middle;
    }

    .users-table tr:hover {
        background: rgba(99, 102, 241, 0.05);
    }

    .user-avatar {
        width: 50px;
        height: 50px;
        border-radius: 12px;
        object-fit: cover;
        border: 2px solid var(--gray-light);
    }

    .user-info {
        display: flex;
        align-items: center;
        gap: 1rem;
    }

    .user-details h4 {
        font-weight: 600;
        color: var(--dark);
        margin-bottom: 0.2rem;
    }

    .user-details p {
        color: var(--gray);
        font-size: 0.9rem;
    }

    .status-badge {
        display: inline-block;
        padding: 0.3rem 0.8rem;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .status-active { background: #dcfce7; color: #166534; }
    .status-blocked { background: #fef2f2; color: #dc2626; }
    .status-verified { background: #dbeafe; color: #1d4ed8; }
    .status-pending { background: #fef3c7; color: #d97706; }

    .action-buttons {
        display: flex;
        gap: 0.5rem;
    }

    .btn {
        padding: 0.5rem 1rem;
        border: none;
        border-radius: 8px;
        font-size: 0.8rem;
        font-weight: 600;
        cursor: pointer;
        transition: var(--transition);
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.3rem;
    }

    .btn-primary { background: var(--primary); color: white; }
    .btn-success { background: var(--success); color: white; }
    .btn-danger { background: var(--danger); color: white; }
    .btn-warning { background: var(--warning); color: white; }
    .btn-info { background: var(--info); color: white; }

    .btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
    }

    .alert {
        padding: 1rem 1.5rem;
        border-radius: 12px;
        margin-bottom: 2rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    .alert-success {
        background: #dcfce7;
        color: #166534;
        border: 1px solid #bbf7d0;
    }

    .alert-error {
        background: #fef2f2;
        color: #dc2626;
        border: 1px solid #fecaca;
    }

    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 0.5rem;
        padding: 2rem;
    }

    .pagination a, .pagination span {
        padding: 0.5rem 1rem;
        border-radius: 8px;
        text-decoration: none;
        color: var(--gray);
        font-weight: 600;
        transition: var(--transition);
    }

    .pagination a:hover {
        background: var(--primary);
        color: white;
    }

    .pagination .current {
        background: var(--primary);
        color: white;
    }

    .empty-state {
        text-align: center;
        padding: 4rem 2rem;
        color: var(--gray);
    }

    .empty-state i {
        font-size: 4rem;
        margin-bottom: 1rem;
        color: var(--gray-light);
    }

    .empty-state h3 {
        font-size: 1.5rem;
        margin-bottom: 0.5rem;
        color: var(--dark);
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .main-content {
            padding: 1.5rem;
        }
    }

    @media (max-width: 900px) {
        .main-content {
            margin-left: 0;
            padding: 1rem;
        }
        .admin-container {
            flex-direction: column;
        }
        .filters-row {
            flex-direction: column;
            align-items: stretch;
        }
        .search-box {
            min-width: auto;
        }
        .users-table {
            font-size: 0.9rem;
        }
        .users-table th,
        .users-table td {
            padding: 0.8rem 0.5rem;
        }
    }

    @media (max-width: 600px) {
        .main-content {
            padding: 0.75rem;
        }
        .page-header {
            padding: 1.5rem;
        }
        .page-title {
            font-size: 1.5rem;
        }
        .stats-grid {
            grid-template-columns: 1fr 1fr;
        }
        .users-table-container {
            overflow-x: auto;
        }
        .action-buttons {
            flex-direction: column;
        }
    }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include '../sidebar_new.php'; ?>
        <main class="main-content">
            
            <!-- Page Header -->
            <div class="page-header">
                <h1 class="page-title">
                    <i class="fas fa-users"></i>
                    Users Management
                </h1>
                <p class="page-subtitle">Manage and monitor user accounts on your social media platform</p>
            </div>

            <!-- Alert Messages -->
            <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?>">
                    <i class="fas fa-<?= $messageType === 'success' ? 'check-circle' : 'exclamation-triangle' ?>"></i>
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php endif; ?>

            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($totalUsers) ?></div>
                    <div class="stat-label">Total Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($activeUsers) ?></div>
                    <div class="stat-label">Active Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($verifiedUsers) ?></div>
                    <div class="stat-label">Verified Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number"><?= number_format($blockedUsers) ?></div>
                    <div class="stat-label">Blocked Users</div>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="filters-section">
                <form method="GET" class="filters-row">
                    <div class="search-box">
                        <i class="fas fa-search search-icon"></i>
                        <input 
                            type="text" 
                            name="search" 
                            class="search-input" 
                            placeholder="Search users by name or email..."
                            value="<?= htmlspecialchars($search) ?>"
                        >
                    </div>
                    <select name="filter" class="filter-select">
                        <option value="all" <?= $filter === 'all' ? 'selected' : '' ?>>All Users</option>
                        <option value="active" <?= $filter === 'active' ? 'selected' : '' ?>>Active Only</option>
                        <option value="verified" <?= $filter === 'verified' ? 'selected' : '' ?>>Verified Only</option>
                        <option value="blocked" <?= $filter === 'blocked' ? 'selected' : '' ?>>Blocked Only</option>
                    </select>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter"></i>
                        Filter
                    </button>
                </form>
            </div>

            <!-- Users Table -->
            <div class="users-table-container">
                <div class="table-header">
                    <h3 class="table-title">Users List</h3>
                    <span class="text-muted"><?= count($users) ?> users found</span>
                </div>
                
                <?php if (empty($users)): ?>
                    <div class="empty-state">
                        <i class="fas fa-users"></i>
                        <h3>No users found</h3>
                        <p>Try adjusting your search criteria or filters</p>
                    </div>
                <?php else: ?>
                    <table class="users-table">
                        <thead>
                            <tr>
                                <th>User</th>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Followers</th>
                                <th>Posts</th>
                                <th>Joined</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td>
                                        <div class="user-info">
                                            <img 
                                                src="<?= htmlspecialchars($user['photo_url'] ?? 'https://ui-avatars.com/api/?name=' . urlencode($user['display_name'] ?? 'User') . '&background=6366f1&color=fff&rounded=true&size=50') ?>" 
                                                alt="User Avatar" 
                                                class="user-avatar"
                                            >
                                            <div class="user-details">
                                                <h4><?= htmlspecialchars($user['display_name'] ?? 'Unknown User') ?></h4>
                                                <p>ID: <?= htmlspecialchars($user['id']) ?></p>
                                            </div>
                                        </div>
                                    </td>
                                    <td><?= htmlspecialchars($user['email'] ?? 'N/A') ?></td>
                                    <td>
                                        <?php
                                        $statuses = [];
                                        if ($user['isVerified'] ?? false) {
                                            $statuses[] = '<span class="status-badge status-verified">Verified</span>';
                                        }
                                        if ($user['isBlocked'] ?? false) {
                                            $statuses[] = '<span class="status-badge status-blocked">Blocked</span>';
                                        } else {
                                            $statuses[] = '<span class="status-badge status-active">Active</span>';
                                        }
                                        echo implode(' ', $statuses);
                                        ?>
                                    </td>
                                    <td><?= number_format($user['followeresCount'] ?? 0) ?></td>
                                    <td><?= number_format($user['postsCount'] ?? 0) ?></td>
                                    <td>
                                        <?php
                                        if (isset($user['created_time'])) {
                                            $timestamp = $user['created_time']->getTimestamp();
                                            echo date('M j, Y', $timestamp);
                                        } else {
                                            echo 'N/A';
                                        }
                                        ?>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="user_details_firebase.php?id=<?= $user['id'] ?>" class="btn btn-info">
                                                <i class="fas fa-eye"></i>
                                                View
                                            </a>
                                            
                                            <?php if ($user['isBlocked'] ?? false): ?>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="unblock_user">
                                                    <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                    <button type="submit" class="btn btn-success" onclick="return confirm('Unblock this user?')">
                                                        <i class="fas fa-unlock"></i>
                                                        Unblock
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="block_user">
                                                    <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                    <button type="submit" class="btn btn-warning" onclick="return confirm('Block this user?')">
                                                        <i class="fas fa-ban"></i>
                                                        Block
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                            
                                            <?php if (!($user['isVerified'] ?? false)): ?>
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="verify_user">
                                                    <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                    <button type="submit" class="btn btn-primary" onclick="return confirm('Verify this user?')">
                                                        <i class="fas fa-check"></i>
                                                        Verify
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                            
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="delete_user">
                                                <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                                <button type="submit" class="btn btn-danger" onclick="return confirm('Delete this user permanently? This action cannot be undone.')">
                                                    <i class="fas fa-trash"></i>
                                                    Delete
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>

            <!-- Pagination -->
            <?php if (count($users) >= $limit): ?>
                <div class="pagination">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&filter=<?= urlencode($filter) ?>">
                            <i class="fas fa-chevron-left"></i> Previous
                        </a>
                    <?php endif; ?>
                    
                    <span class="current">Page <?= $page ?></span>
                    
                    <a href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&filter=<?= urlencode($filter) ?>">
                        Next <i class="fas fa-chevron-right"></i>
                    </a>
                </div>
            <?php endif; ?>
        </main>
    </div>
    
    <script>
    // Auto-submit form on filter change
    document.querySelector('.filter-select').addEventListener('change', function() {
        this.form.submit();
    });
    
    // Add loading state to buttons
    document.querySelectorAll('form').forEach(form => {
        form.addEventListener('submit', function() {
            const button = this.querySelector('button[type="submit"]');
            if (button) {
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
                button.disabled = true;
            }
        });
    });
    </script>
</body>
</html>
