# Firebase Collections Schema for Postery Social Media App

## Overview
This document outlines the Firebase Firestore collections structure for the Postery social media application admin panel.

## Collections Structure

### 1. users
**Description**: User profiles and account information
```javascript
{
  email: "string",
  display_name: "string", 
  photo_url: "string (image path)",
  uid: "string",
  created_time: "timestamp",
  phone_number: "string",
  bio: "string",
  followers: ["array of user document references"],
  following: ["array of user document references"],
  followeresCount: "number",
  lastSeen: "timestamp",
  isOnline: "boolean",
  viewedPosts: ["array of post document references"],
  isVerified: "boolean",
  isEligible: "boolean",
  blockedUsers: ["array of user document references"],
  balance: "number",
  isBlocked: "boolean",
  isActive: "boolean",
  lastLoginAt: "timestamp",
  deviceTokens: ["array of strings"],
  privacySettings: {
    profileVisibility: "string",
    showOnlineStatus: "boolean",
    allowMessages: "boolean"
  }
}
```

### 2. posts
**Description**: User posts and content
```javascript
{
  postImage: "string (image path)",
  postUser: "document reference (users)",
  usersLikes: "number",
  usersShares: "number", 
  postText: "string",
  postUserName: "string",
  postUserImage: "string (image path)",
  createdOn: "timestamp",
  likedBy: ["array of user document references"],
  hashtags: ["array of strings"],
  view: ["array of user document references"],
  isActive: "boolean",
  isReported: "boolean",
  reportCount: "number",
  location: {
    latitude: "number",
    longitude: "number",
    address: "string"
  },
  postType: "string (text/image/video)",
  mediaUrls: ["array of strings"],
  commentsCount: "number"
}
```

### 3. comments
**Description**: Comments on posts
```javascript
{
  commentUser: "document reference (users)",
  commentText: "string",
  commentLikes: "number",
  postRef: "document reference (posts)",
  commentTime: "timestamp",
  isActive: "boolean",
  isReported: "boolean",
  reportCount: "number",
  parentCommentId: "string (for replies)",
  repliesCount: "number"
}
```

### 4. stories
**Description**: User stories (24-hour content)
```javascript
{
  image: "string (image path)",
  uploadedBy: "document reference (users)",
  viewedBy: ["array of user document references"],
  uploadedOn: "timestamp",
  caption: "string",
  url: "string",
  backgroundColor: "string (color)",
  durationSeconds: "number",
  likeBy: ["array of user document references"],
  isActive: "boolean",
  expiresAt: "timestamp",
  viewCount: "number"
}
```

### 5. chats
**Description**: Chat conversations
```javascript
{
  participants: ["array of user document references"],
  lastMessage: "string",
  lastMessageTime: "timestamp",
  lastMessageBy: "document reference (users)",
  isActive: "boolean",
  chatType: "string (private/group)",
  groupName: "string (for group chats)",
  groupImage: "string (image path)",
  createdBy: "document reference (users)",
  createdAt: "timestamp"
}
```

### 6. messages
**Description**: Individual messages in chats (subcollection of chats)
```javascript
{
  sender: "document reference (users)",
  message: "string",
  messageType: "string (text/image/video/audio)",
  mediaUrl: "string",
  timestamp: "timestamp",
  isRead: "boolean",
  readBy: ["array of user document references"],
  isDeleted: "boolean"
}
```

### 7. Reports
**Description**: User reports and moderation
```javascript
{
  postRef: "document reference (posts)",
  reportTime: "timestamp",
  resolved: "boolean",
  resolvedStatement: "string",
  reportedBy: "document reference (users)",
  storyRef: "document reference (stories)",
  userRef: "document reference (users)",
  reportType: "string (spam/harassment/inappropriate/etc)",
  reportReason: "string",
  resolvedBy: "document reference (users)",
  resolvedAt: "timestamp",
  priority: "string (low/medium/high)"
}
```

### 8. notifications
**Description**: Push notifications and in-app notifications
```javascript
{
  userId: "document reference (users)",
  title: "string",
  body: "string",
  type: "string (like/comment/follow/post/etc)",
  data: "object",
  isRead: "boolean",
  createdAt: "timestamp",
  readAt: "timestamp",
  actionUrl: "string",
  imageUrl: "string"
}
```

### 9. CustomAds
**Description**: Custom advertisement management
```javascript
{
  company_name: "string",
  caption: "string",
  call_to_action_url: "string",
  is_active: "boolean",
  created_at: "timestamp",
  id: "string",
  company_avatar_url: "string (image path)",
  ad_creative_url: "string (image path)",
  targetAudience: "object",
  budget: "number",
  impressions: "number",
  clicks: "number",
  startDate: "timestamp",
  endDate: "timestamp"
}
```

### 10. withdrawals
**Description**: User withdrawal requests
```javascript
{
  userRef: "document reference (users)",
  amount: "number",
  status: "string (pending/approved/rejected)",
  created_at: "timestamp",
  withdraw_method: "string",
  accountDetails: "object",
  processedBy: "document reference (users)",
  processedAt: "timestamp",
  rejectionReason: "string",
  transactionId: "string"
}
```

### 11. monetization_config
**Description**: App monetization settings
```javascript
{
  min_followers_to_apply: "number",
  min_posts_to_apply: "number", 
  ad_frequency: "number",
  earning_rate_per_view: "number",
  min_withdrawal_amount: "number",
  max_withdrawal_amount: "number",
  withdrawal_fee_percentage: "number",
  isMonetizationEnabled: "boolean",
  eligibleCountries: ["array of strings"],
  paymentMethods: ["array of strings"]
}
```

### 12. earnings
**Description**: User earnings tracking
```javascript
{
  userRef: "document reference (users)",
  totalEarnings: "number",
  availableBalance: "number",
  totalWithdrawn: "number",
  lastUpdated: "timestamp",
  earningsBreakdown: {
    postViews: "number",
    adViews: "number",
    referrals: "number",
    bonuses: "number"
  },
  monthlyEarnings: "object",
  isEligible: "boolean"
}
```

### 13. monetizationRequests
**Description**: User requests to join monetization program
```javascript
{
  userRef: "document reference (users)",
  status: "string (pending/approved/rejected)",
  created_on: "timestamp",
  reviewedBy: "document reference (users)",
  reviewedAt: "timestamp",
  rejectionReason: "string",
  userDetails: "object",
  documents: ["array of strings"],
  notes: "string"
}
```

### 14. app_settings
**Description**: Global app configuration
```javascript
{
  native_ad_On: "boolean",
  banner_ad_On: "boolean",
  interstitial_ad_On: "boolean",
  reward_ad_On: "boolean",
  custom_ad_frequency: "number",
  native_ad_frequency: "number",
  custom_ad_On: "boolean",
  native_startAfterIndex: "number",
  admobBannerAdUnit: "string",
  admobNativeAdUnit: "string",
  admobInterAdUnit: "string",
  admobRewardAdUnit: "string",
  aboutUs: "string",
  privacyPolicy: "string",
  termsConditions: "string",
  supportEmail: "string",
  communityGuidelines: "string",
  maintenanceMode: "boolean",
  appVersion: "string",
  forceUpdate: "boolean",
  minAppVersion: "string"
}
```

### 15. userFeedbacks
**Description**: User feedback and suggestions
```javascript
{
  userId: "document reference (users)",
  category: "string",
  title: "string",
  feedbackText: "string",
  rating: "number",
  imageUrl: "string (image path)",
  timestamp: "timestamp",
  status: "string (new/in_progress/resolved)",
  priority: "string (low/medium/high)",
  assignedTo: "document reference (users)",
  response: "string",
  respondedAt: "timestamp"
}
```

### 16. reportProblems
**Description**: Technical problem reports
```javascript
{
  userId: "document reference (users)",
  problemType: "string",
  title: "string",
  description: "string",
  imageUrl: "string (image path)",
  timestamp: "timestamp",
  status: "string (new/in_progress/resolved)",
  priority: "string (low/medium/high)",
  deviceInfo: "object",
  appVersion: "string",
  osVersion: "string",
  assignedTo: "document reference (users)",
  resolution: "string",
  resolvedAt: "timestamp"
}
```

### 17. deleteAccountRequests
**Description**: Account deletion requests
```javascript
{
  userRef: "document reference (users)",
  date: "timestamp",
  reason: "string",
  status: "string (pending/approved/rejected)",
  processedBy: "document reference (users)",
  processedAt: "timestamp",
  dataRetentionPeriod: "number",
  notes: "string"
}
```

## Indexes Required

### Composite Indexes
1. **posts**: `createdOn` (desc), `isActive` (asc)
2. **comments**: `postRef` (asc), `commentTime` (desc)
3. **stories**: `uploadedOn` (desc), `isActive` (asc)
4. **notifications**: `userId` (asc), `createdAt` (desc)
5. **reports**: `resolved` (asc), `reportTime` (desc)
6. **withdrawals**: `status` (asc), `created_at` (desc)
7. **earnings**: `userRef` (asc), `lastUpdated` (desc)

### Single Field Indexes
- All foreign key references (userRef, postRef, etc.)
- Timestamp fields for sorting
- Status fields for filtering
- Boolean fields for filtering

## Security Rules (Firestore)

```javascript
rules_version = '2';
service cloud.firestore {
  match /databases/{database}/documents {
    // Users can read/write their own data
    match /users/{userId} {
      allow read, write: if request.auth != null && request.auth.uid == userId;
    }
    
    // Posts are readable by all authenticated users
    match /posts/{postId} {
      allow read: if request.auth != null;
      allow write: if request.auth != null && 
        (resource == null || resource.data.postUser == request.auth.uid);
    }
    
    // Comments are readable by all, writable by authenticated users
    match /comments/{commentId} {
      allow read: if request.auth != null;
      allow create: if request.auth != null;
      allow update, delete: if request.auth != null && 
        resource.data.commentUser == request.auth.uid;
    }
    
    // Admin-only collections
    match /reports/{reportId} {
      allow read, write: if request.auth != null && 
        get(/databases/$(database)/documents/users/$(request.auth.uid)).data.isAdmin == true;
    }
    
    match /app_settings/{settingId} {
      allow read: if request.auth != null;
      allow write: if request.auth != null && 
        get(/databases/$(database)/documents/users/$(request.auth.uid)).data.isAdmin == true;
    }
  }
}
```

## Storage Rules

```javascript
rules_version = '2';
service firebase.storage {
  match /b/{bucket}/o {
    // User profile images
    match /users/{userId}/profile/{fileName} {
      allow read: if request.auth != null;
      allow write: if request.auth != null && request.auth.uid == userId;
    }
    
    // Post images
    match /posts/{postId}/{fileName} {
      allow read: if request.auth != null;
      allow write: if request.auth != null;
    }
    
    // Story images
    match /stories/{storyId}/{fileName} {
      allow read: if request.auth != null;
      allow write: if request.auth != null;
    }
    
    // Admin uploads
    match /admin/{fileName} {
      allow read: if request.auth != null;
      allow write: if request.auth != null && 
        firestore.get(/databases/(default)/documents/users/$(request.auth.uid)).data.isAdmin == true;
    }
  }
}
```
