<?php
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/token.php';

header('Content-Type: application/json');
$admin = require_auth();

// User stats
$totalUsers = $pdo->query('SELECT COUNT(*) FROM users')->fetchColumn();

// Active Users: Count users who completed tasks today and yesterday
$activeUsers = $pdo->query('SELECT COUNT(DISTINCT tc.user_id) FROM task_completions tc 
    WHERE DATE(tc.completed_at) >= DATE_SUB(CURDATE(), INTERVAL 1 DAY) 
    AND tc.user_id IN (SELECT id FROM users WHERE is_blocked = 0)')->fetchColumn();

$blockedUsers = $pdo->query('SELECT COUNT(*) FROM users WHERE is_blocked=1')->fetchColumn();
$totalCoins = $pdo->query('SELECT SUM(coins) FROM users')->fetchColumn();

// Coin distribution (for bar chart)
$coinDistribution = $pdo->query('SELECT coins FROM users')->fetchAll(PDO::FETCH_COLUMN);

// Daily check-in trends (last 14 days)
$checkinTrends = $pdo->query('SELECT DATE(claimed_at) as date, COUNT(*) as count FROM daily_checkins WHERE claimed_at >= DATE_SUB(CURDATE(), INTERVAL 13 DAY) GROUP BY date ORDER BY date ASC')->fetchAll();

// Task completion rates by type (last 14 days)
$taskCompletionTrends = $pdo->query('SELECT by_task, DATE(completed_at) as date, COUNT(*) as count FROM task_completions WHERE completed_at >= DATE_SUB(CURDATE(), INTERVAL 13 DAY) GROUP BY by_task, date ORDER BY date ASC')->fetchAll();

// Task completion rates by user (top 10)
$taskCompletionByUser = $pdo->query('SELECT user_id, COUNT(*) as count FROM task_completions GROUP BY user_id ORDER BY count DESC LIMIT 10')->fetchAll();

// Referral program performance
$referralStats = $pdo->query('SELECT COUNT(*) as total_referrals, SUM(referrer_coins) as total_referrer_coins, SUM(referred_coins) as total_referred_coins FROM referral_rewards')->fetch(PDO::FETCH_ASSOC);

// Redemption analytics: trends by category/subcategory (last 14 days)
$redemptionTrends = $pdo->query('SELECT ros.category_id, roc.category_name, DATE(rh.created_at) as date, COUNT(*) as count FROM redeem_history rh JOIN redeem_options_subcategories ros ON rh.subcategory_id=ros.id JOIN redeem_options_categories roc ON ros.category_id=roc.id WHERE rh.created_at >= DATE_SUB(CURDATE(), INTERVAL 13 DAY) GROUP BY ros.category_id, roc.category_name, date ORDER BY date ASC')->fetchAll();

// Popular redemption options (top 5)
$popularRedemptions = $pdo->query('SELECT ros.sub_category_name, COUNT(*) as count FROM redeem_history rh JOIN redeem_options_subcategories ros ON rh.subcategory_id=ros.id GROUP BY ros.sub_category_name ORDER BY count DESC LIMIT 5')->fetchAll();

// Coin-to-cash conversion rates (average per redemption)
$coinToCash = $pdo->query('SELECT AVG(ros.coins/ros.cash) as avg_rate FROM redeem_history rh JOIN redeem_options_subcategories ros ON rh.subcategory_id=ros.id')->fetchColumn();

// Redemption status breakdown (pie chart)
$redemptionStatus = $pdo->query('SELECT status, COUNT(*) as count FROM redeem_history GROUP BY status')->fetchAll();

// Task engagement: user_task_usage (last 14 days)
$taskUsageTrends = $pdo->query('SELECT task_type, usage_date, SUM(used_count) as total_used FROM user_task_usage WHERE usage_date >= DATE_SUB(CURDATE(), INTERVAL 13 DAY) GROUP BY task_type, usage_date ORDER BY usage_date ASC')->fetchAll();

// Promo code performance
$promoCodeStats = $pdo->query('SELECT pc.code, pc.usage_limit, pc.expiry_date, COUNT(pcu.id) as used_count FROM promo_codes pc LEFT JOIN promo_code_usage pcu ON pc.id=pcu.promo_code_id GROUP BY pc.id ORDER BY used_count DESC')->fetchAll();

// Ad performance: ad_controls and ads_settings
$adControls = $pdo->query('SELECT * FROM ad_controls')->fetchAll();
$adSettings = $pdo->query('SELECT * FROM ads_settings')->fetchAll();

// Recent activity - diverse activities from multiple sources
$recentActivity = [];

// Recent user registrations (last 3)
$recentUsers = $pdo->query('SELECT email, created_at FROM users ORDER BY created_at DESC LIMIT 3')->fetchAll();
foreach ($recentUsers as $row) {
    $recentActivity[] = [
        'icon' => 'fa-user-plus',
        'title' => 'New user registered',
        'description' => 'User ' . $row['email'] . ' joined the app',
        'time' => $row['created_at'],
        'type' => 'user_registration'
    ];
}

// Recent redeem requests (last 3)
$recentRedeems = $pdo->query('SELECT rh.*, u.email, ros.sub_category_name FROM redeem_history rh 
    JOIN users u ON rh.user_id = u.id 
    JOIN redeem_options_subcategories ros ON rh.subcategory_id = ros.id 
    ORDER BY rh.created_at DESC LIMIT 3')->fetchAll();
foreach ($recentRedeems as $row) {
    $statusIcon = $row['status'] === 'approved' ? 'fa-check-circle' : ($row['status'] === 'rejected' ? 'fa-times-circle' : 'fa-clock');
    $statusColor = $row['status'] === 'approved' ? 'text-success' : ($row['status'] === 'rejected' ? 'text-danger' : 'text-warning');
    $recentActivity[] = [
        'icon' => 'fa-gift',
        'title' => 'Redemption request',
        'description' => 'User ' . $row['email'] . ' requested ' . $row['sub_category_name'] . ' (Status: ' . ucfirst($row['status']) . ')',
        'time' => $row['created_at'],
        'type' => 'redemption',
        'status' => $row['status']
    ];
}

// Recent task completions (last 3)
$recentTasks = $pdo->query('SELECT tc.*, u.email FROM task_completions tc 
    JOIN users u ON tc.user_id = u.id 
    ORDER BY tc.completed_at DESC LIMIT 3')->fetchAll();
foreach ($recentTasks as $row) {
    $recentActivity[] = [
        'icon' => 'fa-tasks',
        'title' => 'Task completed',
        'description' => 'User ' . $row['email'] . ' completed ' . $row['by_task'] . ' task (+' . $row['earned_coins'] . ' coins)',
        'time' => $row['completed_at'],
        'type' => 'task_completion'
    ];
}

// Recent daily check-ins (last 3)
$recentCheckins = $pdo->query('SELECT dc.*, u.email FROM daily_checkins dc 
    JOIN users u ON dc.user_id = u.id 
    ORDER BY dc.claimed_at DESC LIMIT 3')->fetchAll();
foreach ($recentCheckins as $row) {
    $recentActivity[] = [
        'icon' => 'fa-calendar-check',
        'title' => 'Daily check-in',
        'description' => 'User ' . $row['email'] . ' claimed daily bonus (+' . $row['coins_earned'] . ' coins)',
        'time' => $row['claimed_at'],
        'type' => 'daily_checkin'
    ];
}

// Recent promo code usage (last 3)
$recentPromos = $pdo->query('SELECT pcu.*, u.email, pc.code FROM promo_code_usage pcu 
    JOIN users u ON pcu.user_id = u.id 
    JOIN promo_codes pc ON pcu.promo_code_id = pc.id 
    ORDER BY pcu.created_at DESC LIMIT 3')->fetchAll();
foreach ($recentPromos as $row) {
    $recentActivity[] = [
        'icon' => 'fa-tag',
        'title' => 'Promo code used',
        'description' => 'User ' . $row['email'] . ' used promo code ' . $row['code'],
        'time' => $row['created_at'],
        'type' => 'promo_usage'
    ];
}

// Sort all activities by time (most recent first) and take top 8
usort($recentActivity, function($a, $b) {
    return strtotime($b['time']) - strtotime($a['time']);
});
$recentActivity = array_slice($recentActivity, 0, 8);

// Output all analytics

echo json_encode([
    'user_stats' => [
        'total_users' => (int)$totalUsers,
        'active_users' => (int)$activeUsers,
        'blocked_users' => (int)$blockedUsers,
        'total_coins' => (int)$totalCoins,
        'coin_distribution' => $coinDistribution
    ],
    'checkin_trends' => $checkinTrends,
    'task_completion_trends' => $taskCompletionTrends,
    'task_completion_by_user' => $taskCompletionByUser,
    'referral_stats' => $referralStats,
    'redemption_trends' => $redemptionTrends,
    'popular_redemptions' => $popularRedemptions,
    'coin_to_cash' => $coinToCash,
    'redemption_status' => $redemptionStatus,
    'task_usage_trends' => $taskUsageTrends,
    'promo_code_stats' => $promoCodeStats,
    'ad_controls' => $adControls,
    'ad_settings' => $adSettings,
    'recent_activity' => $recentActivity
]); 