<?php
// Use production Firebase configuration
require_once __DIR__ . '/firebase_production.php';

header('Content-Type: application/json');
require_firebase_auth();

// OneSignal credentials (replace with your real keys)
define('ONESIGNAL_APP_ID', 'YOUR_ONESIGNAL_APP_ID');
define('ONESIGNAL_API_KEY', 'YOUR_ONESIGNAL_API_KEY');

function send_onesignal_notification($title, $message, $image_url = null, $action_url = null, $platform = 'all', $schedule = null, $buttons = []) {
    $content = ["en" => $message];
    $headings = ["en" => $title];
    $fields = [
        'app_id' => ONESIGNAL_APP_ID,
        'headings' => $headings,
        'contents' => $content,
        'included_segments' => ['All']
    ];
    
    // Platform filter
    if ($platform && $platform !== 'all') {
        unset($fields['included_segments']);
        $fields['filters'] = [
            ["field" => "device_type", "relation" => "=", "value" => (
                $platform === 'android' ? 1 : ($platform === 'ios' ? 0 : 5)
            )]
        ];
    }
    
    // Image
    if ($image_url) {
        $fields['big_picture'] = $image_url;
        $fields['ios_attachments'] = ["id1" => $image_url];
        $fields['chrome_web_image'] = $image_url;
    }
    
    // Action URL
    if ($action_url) {
        $fields['url'] = $action_url;
    }
    
    // Schedule
    if ($schedule) {
        $fields['send_after'] = date('c', strtotime($schedule));
    }
    
    // Buttons
    if (!empty($buttons)) {
        $fields['buttons'] = $buttons;
    }
    
    $fields = json_encode($fields);
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://onesignal.com/api/v1/notifications');
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json; charset=utf-8',
        'Authorization: Basic ' . ONESIGNAL_API_KEY
    ]);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $fields);
    $result = curl_exec($ch);
    $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $res = json_decode($result, true);
    
    if ($httpcode === 200 && isset($res['id'])) {
        return ['success'=>true, 'message'=>'Notification sent!', 'onesignal_id'=>$res['id'], 'user_count'=>$res['recipients'] ?? 0];
    } else {
        return ['success'=>false, 'message'=>'Failed to send: '.($res['errors'][0]??'Unknown error')];
    }
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    $limit = $_GET['limit'] ?? 50;
    $orderBy = $_GET['orderBy'] ?? 'created_at';
    $direction = $_GET['direction'] ?? 'desc';
    $status = $_GET['status'] ?? ''; // all, sent, pending, failed, scheduled
    
    $where = [];
    if ($status && $status !== 'all') {
        $where['status'] = $status;
    }
    
    $notifications = FirebaseHelper::getCollection('notifications', $limit, ['field' => $orderBy, 'direction' => $direction], $where);
    
    echo json_encode(['success' => true, 'data' => $notifications]);
    exit;
}

if ($method === 'GET' && isset($_GET['id'])) {
    $notificationId = $_GET['id'];
    $notification = FirebaseHelper::getDocument('notifications', $notificationId);
    
    if ($notification) {
        echo json_encode(['success' => true, 'data' => $notification]);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Notification not found']);
    }
    exit;
}

if ($method === 'POST') {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $title = trim($data['title'] ?? '');
    $message = trim($data['message'] ?? '');
    $image_url = trim($data['image_url'] ?? '');
    $action_url = trim($data['action_url'] ?? '');
    $platform = $data['platform'] ?? 'all';
    $schedule = $data['schedule'] ?? null;
    
    // Buttons
    $buttons = [];
    $button1_text = $data['button1_text'] ?? '';
    $button1_url = $data['button1_url'] ?? '';
    $button2_text = $data['button2_text'] ?? '';
    $button2_url = $data['button2_url'] ?? '';
    
    if (!empty($button1_text) && !empty($button1_url)) {
        $buttons[] = [
            'id' => 'btn1',
            'text' => $button1_text,
            'url' => $button1_url
        ];
    }
    if (!empty($button2_text) && !empty($button2_url)) {
        $buttons[] = [
            'id' => 'btn2',
            'text' => $button2_text,
            'url' => $button2_url
        ];
    }
    
    if (!$title || !$message) {
        http_response_code(400);
        echo json_encode(['success'=>false,'message'=>'Title and message required']);
        exit;
    }
    
    // Store notification in Firebase first
    $notificationData = [
        'title' => $title,
        'message' => $message,
        'image_url' => $image_url,
        'action_url' => $action_url,
        'platform' => $platform,
        'schedule_time' => $schedule ? date('Y-m-d H:i:s', strtotime($schedule)) : null,
        'status' => $schedule ? 'scheduled' : 'pending',
        'button1_text' => $button1_text,
        'button1_url' => $button1_url,
        'button2_text' => $button2_text,
        'button2_url' => $button2_url,
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s')
    ];
    
    $notificationId = FirebaseHelper::addDocument('notifications', $notificationData);
    
    if (!$notificationId) {
        http_response_code(500);
        echo json_encode(['success'=>false,'message'=>'Failed to save notification']);
        exit;
    }
    
    // If scheduled, just store and return
    if ($schedule) {
        echo json_encode(['success'=>true, 'message'=>'Notification scheduled for ' . date('d-m-Y H:i', strtotime($schedule)), 'id' => $notificationId]);
        exit;
    }
    
    // Send notification immediately
    $result = send_onesignal_notification($title, $message, $image_url, $action_url, $platform, null, $buttons);
    
    if ($result['success']) {
        // Update notification with sent details
        FirebaseHelper::updateDocument('notifications', $notificationId, [
            'status' => 'sent',
            'sent_time' => date('Y-m-d H:i:s'),
            'user_count' => $result['user_count'],
            'onesignal_id' => $result['onesignal_id']
        ]);
        
        echo json_encode(['success'=>true, 'message'=>'Notification sent to ' . $result['user_count'] . ' users!', 'id' => $notificationId]);
    } else {
        // Update notification with failed status
        FirebaseHelper::updateDocument('notifications', $notificationId, [
            'status' => 'failed',
            'error_message' => $result['message']
        ]);
        
        echo json_encode($result);
    }
    exit;
}

if ($method === 'PUT') {
    $notificationId = $_GET['id'] ?? null;
    if (!$notificationId) {
        http_response_code(400);
        echo json_encode(['error' => 'Notification ID required']);
        exit;
    }
    
    $data = json_decode(file_get_contents('php://input'), true);
    
    $updateData = [];
    $allowedFields = [
        'title', 'message', 'image_url', 'action_url', 'platform', 'schedule_time',
        'button1_text', 'button1_url', 'button2_text', 'button2_url'
    ];
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $updateData[$field] = $data[$field];
        }
    }
    
    if (empty($updateData)) {
        http_response_code(400);
        echo json_encode(['error' => 'No fields to update']);
        exit;
    }
    
    $success = FirebaseHelper::updateDocument('notifications', $notificationId, $updateData);
    
    if ($success) {
        echo json_encode(['success' => true, 'message' => 'Notification updated successfully']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to update notification']);
    }
    exit;
}

if ($method === 'DELETE') {
    $notificationId = $_GET['id'] ?? null;
    if (!$notificationId) {
        http_response_code(400);
        echo json_encode(['error' => 'Notification ID required']);
        exit;
    }
    
    $success = FirebaseHelper::deleteDocument('notifications', $notificationId);
    
    if ($success) {
        echo json_encode(['success' => true, 'message' => 'Notification deleted successfully']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to delete notification']);
    }
    exit;
}

// Handle scheduled notifications (cron job endpoint)
if ($method === 'GET' && isset($_GET['cron'])) {
    $scheduledNotifications = FirebaseHelper::getCollection('notifications', null, null, [
        'status' => 'scheduled'
    ]);
    
    $results = [];
    foreach ($scheduledNotifications as $notification) {
        // Check if it's time to send
        if (strtotime($notification['schedule_time']) <= time()) {
            $buttons = [];
            if (!empty($notification['button1_text']) && !empty($notification['button1_url'])) {
                $buttons[] = [
                    'id' => 'btn1',
                    'text' => $notification['button1_text'],
                    'url' => $notification['button1_url']
                ];
            }
            if (!empty($notification['button2_text']) && !empty($notification['button2_url'])) {
                $buttons[] = [
                    'id' => 'btn2',
                    'text' => $notification['button2_text'],
                    'url' => $notification['button2_url']
                ];
            }
            
            $result = send_onesignal_notification(
                $notification['title'], 
                $notification['message'], 
                $notification['image_url'], 
                $notification['action_url'], 
                $notification['platform'], 
                null, 
                $buttons
            );
            
            if ($result['success']) {
                FirebaseHelper::updateDocument('notifications', $notification['id'], [
                    'status' => 'sent',
                    'sent_time' => date('Y-m-d H:i:s'),
                    'user_count' => $result['user_count'],
                    'onesignal_id' => $result['onesignal_id']
                ]);
                $results[] = "Notification #{$notification['id']} sent to {$result['user_count']} users";
            } else {
                FirebaseHelper::updateDocument('notifications', $notification['id'], [
                    'status' => 'failed',
                    'error_message' => $result['message']
                ]);
                $results[] = "Notification #{$notification['id']} failed: " . $result['message'];
            }
        }
    }
    
    echo json_encode(['success'=>true, 'message'=>'Cron job completed', 'results'=>$results]);
    exit;
}

http_response_code(405);
echo json_encode(['error' => 'Method not allowed']);
?>
