<?php
// Simple Installation Script for Postry Social Media Admin Panel
echo "🚀 Postry Social Media Admin Panel - Installation Script\n";
echo "=======================================================\n\n";

$errors = [];
$warnings = [];
$success = [];

// Check PHP version
if (version_compare(PHP_VERSION, '7.4.0') >= 0) {
    $success[] = "PHP version " . PHP_VERSION . " is compatible";
} else {
    $errors[] = "PHP version " . PHP_VERSION . " is too old. Please upgrade to PHP 7.4+";
}

// Check required PHP extensions
$required_extensions = ['curl', 'json', 'mbstring', 'openssl'];
foreach ($required_extensions as $ext) {
    if (extension_loaded($ext)) {
        $success[] = "PHP extension '$ext' is installed";
    } else {
        $errors[] = "PHP extension '$ext' is missing";
    }
}

// Check if service account key exists
if (file_exists(__DIR__ . '/serviceAccountKey.json')) {
    $success[] = "Firebase service account key found";
    
    // Validate JSON
    $keyContent = file_get_contents(__DIR__ . '/serviceAccountKey.json');
    $keyData = json_decode($keyContent, true);
    if ($keyData && isset($keyData['project_id'])) {
        $success[] = "Service account key is valid (Project: " . $keyData['project_id'] . ")";
    } else {
        $errors[] = "Service account key is invalid JSON";
    }
} else {
    $errors[] = "Firebase service account key not found";
}

// Check if composer dependencies are installed
if (file_exists(__DIR__ . '/vendor/autoload.php')) {
    $success[] = "Composer dependencies are installed";
} else {
    $warnings[] = "Composer dependencies not installed. Run: composer install";
}

// Check if .htaccess exists
if (file_exists(__DIR__ . '/.htaccess')) {
    $success[] = ".htaccess file found";
} else {
    $warnings[] = ".htaccess file not found (optional for some servers)";
}

// Check file permissions
$writable_dirs = ['uploads', 'logs'];
foreach ($writable_dirs as $dir) {
    $dir_path = __DIR__ . '/' . $dir;
    if (!file_exists($dir_path)) {
        if (mkdir($dir_path, 0755, true)) {
            $success[] = "Created directory: $dir";
        } else {
            $warnings[] = "Could not create directory: $dir";
        }
    }
    
    if (is_writable($dir_path)) {
        $success[] = "Directory '$dir' is writable";
    } else {
        $warnings[] = "Directory '$dir' is not writable";
    }
}

// Test Firebase connection if possible
if (file_exists(__DIR__ . '/vendor/autoload.php') && file_exists(__DIR__ . '/serviceAccountKey.json')) {
    try {
        require_once __DIR__ . '/vendor/autoload.php';
        require_once __DIR__ . '/firebase_config.php';
        
        $firestore = FirebaseConfig::getFirestore();
        $success[] = "Firebase connection test successful";
    } catch (Exception $e) {
        $errors[] = "Firebase connection failed: " . $e->getMessage();
    }
}

// Display results
echo "📊 Installation Check Results:\n";
echo "==============================\n\n";

if (!empty($success)) {
    echo "✅ SUCCESS:\n";
    foreach ($success as $msg) {
        echo "   • $msg\n";
    }
    echo "\n";
}

if (!empty($warnings)) {
    echo "⚠️  WARNINGS:\n";
    foreach ($warnings as $msg) {
        echo "   • $msg\n";
    }
    echo "\n";
}

if (!empty($errors)) {
    echo "❌ ERRORS:\n";
    foreach ($errors as $msg) {
        echo "   • $msg\n";
    }
    echo "\n";
}

// Final status
if (empty($errors)) {
    echo "🎉 INSTALLATION STATUS: READY!\n";
    echo "===============================\n\n";
    
    echo "🚀 Your admin panel is ready to use!\n\n";
    
    echo "📋 Next Steps:\n";
    echo "1. Create admin user:\n";
    echo "   curl -X POST http://localhost/backend/firebase_auth.php?action=create_admin \\\n";
    echo "     -H \"Content-Type: application/json\" \\\n";
    echo "     -d '{\"email\":\"admin@postry.com\",\"password\":\"admin123\"}'\n\n";
    
    echo "2. Login:\n";
    echo "   curl -X POST http://localhost/backend/firebase_auth.php?action=login \\\n";
    echo "     -H \"Content-Type: application/json\" \\\n";
    echo "     -d '{\"email\":\"admin@postry.com\",\"password\":\"admin123\"}'\n\n";
    
    echo "3. Test dashboard:\n";
    echo "   curl -X GET http://localhost/backend/firebase_dashboard.php \\\n";
    echo "     -H \"Authorization: Bearer YOUR_TOKEN\"\n\n";
    
    echo "📖 For detailed setup instructions, see: QUICK_SETUP_GUIDE.md\n";
    
} else {
    echo "❌ INSTALLATION STATUS: NEEDS ATTENTION\n";
    echo "======================================\n\n";
    
    echo "Please fix the errors above before proceeding.\n\n";
    
    echo "🔧 Common Solutions:\n";
    echo "• Install PHP extensions: sudo apt-get install php-curl php-json php-mbstring\n";
    echo "• Install Composer: https://getcomposer.org/download/\n";
    echo "• Run: composer install\n";
    echo "• Check file permissions\n\n";
}

echo "📞 Need help? Check the documentation files:\n";
echo "• QUICK_SETUP_GUIDE.md\n";
echo "• FIREBASE_SETUP_GUIDE.md\n";
echo "• FIREBASE_MIGRATION_SUMMARY.md\n\n";

echo "🎯 Project: Postry Social Media Admin Panel\n";
echo "🔥 Backend: Firebase Firestore\n";
echo "📱 Ready for: Mobile App Integration\n\n";
?>
