<?php
session_start();
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: pages/login_firebase.php');
    exit;
}

require_once __DIR__ . '/backend/firebase_config.php';

if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: pages/login_firebase.php');
    exit;
}

// Get dashboard statistics from Firebase
try {
    $firestore = FirebaseConfig::getFirestore();
    
    // Get total users count
    $totalUsers = FirebaseHelper::getCollectionCount('users');
    
    // Get active users count
    $activeUsers = FirebaseHelper::getCollectionCount('users', ['isActive' => true]);
    
    // Get total posts count
    $totalPosts = FirebaseHelper::getCollectionCount('posts');
    
    // Get total stories count
    $totalStories = FirebaseHelper::getCollectionCount('stories');
    
    // Get pending reports count
    $pendingReports = FirebaseHelper::getCollectionCount('Reports', ['resolved' => false]);
    
    // Get recent users
    $recentUsers = FirebaseHelper::getCollection('users', 5, ['field' => 'created_time', 'direction' => 'DESC']);
    
    // Get recent posts
    $recentPosts = FirebaseHelper::getCollection('posts', 5, ['field' => 'createdOn', 'direction' => 'DESC']);
    
} catch (Exception $e) {
    error_log('Dashboard data fetch error: ' . $e->getMessage());
    $totalUsers = $activeUsers = $totalPosts = $totalStories = $pendingReports = 0;
    $recentUsers = $recentPosts = [];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Postery Social Media - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
    :root {
        --primary: #6366f1;
        --primary-dark: #4f46e5;
        --secondary: #8b5cf6;
        --success: #10b981;
        --danger: #ef4444;
        --warning: #f59e0b;
        --info: #3b82f6;
        --light: #f8fafc;
        --dark: #1e293b;
        --gray: #64748b;
        --gray-light: #e2e8f0;
        --sidebar-width: 280px;
        --header-height: 70px;
        --card-radius: 12px;
        --transition: all 0.3s ease;
    }

    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
        font-family: 'Inter', 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    }

    body {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: #1e293b;
        overflow-x: hidden;
        line-height: 1.6;
        min-height: 100vh;
    }

    .admin-container {
        display: flex;
        min-height: 100vh;
    }

    .main-content {
        flex: 1;
        padding: 2rem;
        margin-left: var(--sidebar-width);
        transition: var(--transition);
    }

    .welcome-banner {
        background: linear-gradient(135deg, var(--primary), var(--secondary));
        color: white;
        padding: 2.5rem;
        border-radius: var(--card-radius);
        margin-bottom: 2rem;
        box-shadow: 0 10px 30px rgba(99, 102, 241, 0.3);
        position: relative;
        overflow: hidden;
    }

    .welcome-banner::before {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 200%;
        height: 200%;
        background: radial-gradient(circle, rgba(255,255,255,0.1) 0%, transparent 70%);
        animation: float 6s ease-in-out infinite;
    }

    @keyframes float {
        0%, 100% { transform: translateY(0px) rotate(0deg); }
        50% { transform: translateY(-20px) rotate(180deg); }
    }

    .welcome-banner h2 {
        font-size: 2.2rem;
        margin-bottom: 0.5rem;
        font-weight: 700;
        position: relative;
        z-index: 1;
    }

    .welcome-banner p {
        opacity: 0.9;
        font-size: 1.1rem;
        position: relative;
        z-index: 1;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .stat-card {
        background: white;
        border-radius: var(--card-radius);
        padding: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        transition: var(--transition);
        display: flex;
        align-items: center;
        position: relative;
        overflow: hidden;
    }

    .stat-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        width: 4px;
        height: 100%;
        background: var(--primary);
        transform: scaleY(0);
        transition: transform 0.3s ease;
    }

    .stat-card:hover::before {
        transform: scaleY(1);
    }

    .stat-card:hover {
        transform: translateY(-8px);
        box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
    }

    .stat-icon {
        width: 60px;
        height: 60px;
        border-radius: 16px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 1.5rem;
        color: white;
        font-size: 1.5rem;
        position: relative;
    }

    .bg-blue { background: linear-gradient(135deg, #3b82f6, #1d4ed8); }
    .bg-green { background: linear-gradient(135deg, #10b981, #059669); }
    .bg-red { background: linear-gradient(135deg, #ef4444, #dc2626); }
    .bg-orange { background: linear-gradient(135deg, #f59e0b, #d97706); }
    .bg-purple { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
    .bg-pink { background: linear-gradient(135deg, #ec4899, #db2777); }

    .stat-info h3 {
        font-size: 0.9rem;
        color: var(--gray);
        margin-bottom: 0.5rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .stat-info p {
        font-size: 2rem;
        font-weight: 800;
        color: var(--dark);
        margin: 0;
    }

    .dashboard-charts {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
        gap: 2rem;
        margin-bottom: 2rem;
    }

    .chart-card {
        background: white;
        border-radius: var(--card-radius);
        padding: 2rem;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        position: relative;
    }

    .chart-container {
        position: relative;
        height: 300px;
        width: 100%;
    }

    .chart-title {
        font-size: 1.2rem;
        font-weight: 700;
        margin-bottom: 1.5rem;
        color: var(--dark);
        display: flex;
        align-items: center;
    }

    .chart-title i {
        margin-right: 0.5rem;
        color: var(--primary);
    }

    .activity-section {
        margin-bottom: 2rem;
    }

    .section-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 1.5rem;
    }

    .section-header h2 {
        font-size: 1.5rem;
        color: var(--dark);
        font-weight: 700;
    }

    .view-all {
        color: var(--primary);
        text-decoration: none;
        font-size: 0.9rem;
        font-weight: 600;
        transition: var(--transition);
        padding: 0.5rem 1rem;
        border-radius: 8px;
        background: rgba(99, 102, 241, 0.1);
    }

    .view-all:hover {
        background: var(--primary);
        color: white;
        text-decoration: none;
    }

    .activity-list {
        background: white;
        border-radius: var(--card-radius);
        overflow: hidden;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    .activity-item {
        display: flex;
        align-items: center;
        padding: 1.5rem;
        border-bottom: 1px solid var(--gray-light);
        transition: var(--transition);
    }

    .activity-item:last-child {
        border-bottom: none;
    }

    .activity-item:hover {
        background: rgba(99, 102, 241, 0.05);
    }

    .activity-icon {
        width: 50px;
        height: 50px;
        border-radius: 12px;
        background: linear-gradient(135deg, var(--primary), var(--secondary));
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 1rem;
        color: white;
        font-size: 1.2rem;
    }

    .activity-info {
        flex: 1;
    }

    .activity-info h4 {
        font-size: 1rem;
        margin-bottom: 0.3rem;
        color: var(--dark);
        font-weight: 600;
    }

    .activity-info p {
        font-size: 0.9rem;
        color: var(--gray);
    }

    .activity-info a {
        color: var(--primary);
        text-decoration: none;
        font-weight: 600;
    }

    .activity-info a:hover {
        text-decoration: underline;
    }

    .activity-time {
        font-size: 0.8rem;
        color: var(--gray);
        font-weight: 500;
    }

    .quick-actions h2 {
        font-size: 1.5rem;
        color: var(--dark);
        margin-bottom: 1.5rem;
        font-weight: 700;
    }

    .actions-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }

    .action-card {
        background: white;
        border-radius: var(--card-radius);
        padding: 2rem;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        text-align: center;
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
        transition: var(--transition);
        text-decoration: none;
        color: var(--dark);
        position: relative;
        overflow: hidden;
    }

    .action-card::before {
        content: '';
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.4), transparent);
        transition: left 0.5s;
    }

    .action-card:hover::before {
        left: 100%;
    }

    .action-card:hover {
        transform: translateY(-8px);
        box-shadow: 0 12px 40px rgba(0, 0, 0, 0.15);
        color: var(--primary);
    }

    .action-card i {
        font-size: 2.5rem;
        margin-bottom: 1rem;
        color: var(--primary);
        transition: var(--transition);
    }

    .action-card:hover i {
        transform: scale(1.1);
    }

    .action-card span {
        font-size: 1rem;
        font-weight: 600;
    }

    .admin-footer {
        text-align: center;
        padding: 2rem;
        color: var(--gray);
        font-size: 0.9rem;
        background: white;
        border-radius: var(--card-radius);
        box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .main-content {
            padding: 1.5rem;
        }
        .dashboard-charts {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 900px) {
        .main-content {
            margin-left: 0;
            padding: 1rem;
        }
        .admin-container {
            flex-direction: column;
        }
        .stats-grid, .actions-grid {
            grid-template-columns: 1fr 1fr;
        }
        .chart-container {
            height: 250px;
        }
    }

    @media (max-width: 600px) {
        .main-content {
            padding: 0.75rem;
        }
        .welcome-banner {
            padding: 2rem;
        }
        .welcome-banner h2 {
            font-size: 1.8rem;
        }
        .stats-grid, .actions-grid {
            grid-template-columns: 1fr;
        }
        .activity-item {
            flex-direction: column;
            align-items: flex-start;
        }
        .activity-time {
            margin-top: 0.5rem;
            align-self: flex-end;
        }
        .chart-container {
            height: 200px;
        }
        .chart-card {
            padding: 1.5rem;
        }
    }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include 'sidebar_new.php'; ?>
        <main class="main-content">
        
            <!-- Welcome Banner -->
            <div class="welcome-banner">
                <h2>Welcome to Postery Admin Panel! 🚀</h2>
                <p>Manage your social media platform with powerful analytics and insights.</p>
            </div>
            
            <!-- Stats Cards -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon bg-blue">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Total Users</h3>
                        <p><?= number_format($totalUsers) ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon bg-green">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Active Users</h3>
                        <p><?= number_format($activeUsers) ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon bg-purple">
                        <i class="fas fa-images"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Total Posts</h3>
                        <p><?= number_format($totalPosts) ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon bg-orange">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Active Stories</h3>
                        <p><?= number_format($totalStories) ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon bg-red">
                        <i class="fas fa-flag"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Pending Reports</h3>
                        <p><?= number_format($pendingReports) ?></p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon bg-pink">
                        <i class="fas fa-comments"></i>
                    </div>
                    <div class="stat-info">
                        <h3>Total Comments</h3>
                        <p><?= number_format(FirebaseHelper::getCollectionCount('comments')) ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Charts Section -->
            <div class="dashboard-charts">
                <div class="chart-card">
                    <div class="chart-title">
                        <i class="fas fa-chart-line"></i>
                        User Growth Trends
                    </div>
                    <div class="chart-container">
                        <canvas id="userGrowthChart"></canvas>
                    </div>
                </div>
                <div class="chart-card">
                    <div class="chart-title">
                        <i class="fas fa-chart-pie"></i>
                        Content Distribution
                    </div>
                    <div class="chart-container">
                        <canvas id="contentDistributionChart"></canvas>
                    </div>
                </div>
            </div>
            
            <div class="dashboard-charts">
                <div class="chart-card">
                    <div class="chart-title">
                        <i class="fas fa-chart-bar"></i>
                        Engagement Analytics
                    </div>
                    <div class="chart-container">
                        <canvas id="engagementChart"></canvas>
                    </div>
                </div>
                <div class="chart-card">
                    <div class="chart-title">
                        <i class="fas fa-chart-area"></i>
                        Activity Timeline
                    </div>
                    <div class="chart-container">
                        <canvas id="activityTimelineChart"></canvas>
                    </div>
                </div>
            </div>
            
            <!-- Recent Activity -->
            <div class="activity-section">
                <div class="section-header">
                    <h2>Recent Activity</h2>
                    <a href="pages/reports.php" class="view-all">View All Reports</a>
                </div>
                <div class="activity-list">
                    <?php if (!empty($recentUsers)): ?>
                        <?php foreach (array_slice($recentUsers, 0, 3) as $user): ?>
                            <div class="activity-item">
                                <div class="activity-icon">
                                    <i class="fas fa-user-plus"></i>
                                </div>
                                <div class="activity-info">
                                    <h4>New user registered</h4>
                                    <p>User <a href="pages/users_new.php?uid=<?= urlencode($user['id']) ?>"><?= htmlspecialchars($user['display_name'] ?? $user['email']) ?></a> joined the platform</p>
                                </div>
                                <div class="activity-time"><?= isset($user['created_time']) ? date('M j, Y', $user['created_time']->getTimestamp()) : 'Recently' ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    
                    <?php if (!empty($recentPosts)): ?>
                        <?php foreach (array_slice($recentPosts, 0, 2) as $post): ?>
                            <div class="activity-item">
                                <div class="activity-icon">
                                    <i class="fas fa-image"></i>
                                </div>
                                <div class="activity-info">
                                    <h4>New post created</h4>
                                    <p>Post by <a href="pages/users_new.php?uid=<?= urlencode($post['postUser']) ?>"><?= htmlspecialchars($post['postUserName'] ?? 'Unknown User') ?></a></p>
                                </div>
                                <div class="activity-time"><?= isset($post['createdOn']) ? date('M j, Y', $post['createdOn']->getTimestamp()) : 'Recently' ?></div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div class="quick-actions">
                <h2>Quick Actions</h2>
                <div class="actions-grid">
                    <a href="pages/users_new.php" class="action-card">
                        <i class="fas fa-users"></i>
                        <span>Manage Users</span>
                    </a>
                    <a href="pages/posts_new.php" class="action-card">
                        <i class="fas fa-images"></i>
                        <span>Manage Posts</span>
                    </a>
                    <a href="pages/reports.php" class="action-card">
                        <i class="fas fa-flag"></i>
                        <span>Review Reports</span>
                    </a>
                    <a href="pages/analytics.php" class="action-card">
                        <i class="fas fa-chart-line"></i>
                        <span>View Analytics</span>
                    </a>
                    <a href="pages/notifications.php" class="action-card">
                        <i class="fas fa-bell"></i>
                        <span>Send Notifications</span>
                    </a>
                    <a href="pages/settings.php" class="action-card">
                        <i class="fas fa-cog"></i>
                        <span>App Settings</span>
                    </a>
                </div>
            </div>
            
            <!-- Footer -->
            <footer class="admin-footer">
                <p>© <?= date('Y') ?> Postery Social Media. All rights reserved. | Powered by Firebase</p>
            </footer>
        </main>
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize charts with sample data
        initCharts();
    });
    
    function initCharts() {
        // User Growth Chart
        new Chart(document.getElementById('userGrowthChart'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [{
                    label: 'New Users',
                    data: [120, 190, 300, 500, 200, 300],
                    borderColor: '#6366f1',
                    backgroundColor: 'rgba(99, 102, 241, 0.1)',
                    borderWidth: 3,
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: '#fff',
                    pointBorderColor: '#6366f1',
                    pointBorderWidth: 2,
                    pointRadius: 6,
                    pointHoverRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { color: 'rgba(0,0,0,0.05)' }
                    },
                    x: {
                        grid: { display: false }
                    }
                }
            }
        });
        
        // Content Distribution Chart
        new Chart(document.getElementById('contentDistributionChart'), {
            type: 'doughnut',
            data: {
                labels: ['Posts', 'Stories', 'Comments', 'Messages'],
                datasets: [{
                    data: [45, 25, 20, 10],
                    backgroundColor: ['#6366f1', '#8b5cf6', '#10b981', '#f59e0b'],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true
                        }
                    }
                },
                cutout: '60%'
            }
        });
        
        // Engagement Chart
        new Chart(document.getElementById('engagementChart'), {
            type: 'bar',
            data: {
                labels: ['Likes', 'Comments', 'Shares', 'Views'],
                datasets: [{
                    label: 'Engagement',
                    data: [1200, 800, 400, 2000],
                    backgroundColor: ['#6366f1', '#8b5cf6', '#10b981', '#f59e0b'],
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { color: 'rgba(0,0,0,0.05)' }
                    },
                    x: {
                        grid: { display: false }
                    }
                }
            }
        });
        
        // Activity Timeline Chart
        new Chart(document.getElementById('activityTimelineChart'), {
            type: 'line',
            data: {
                labels: ['00:00', '04:00', '08:00', '12:00', '16:00', '20:00'],
                datasets: [{
                    label: 'Active Users',
                    data: [50, 30, 200, 400, 300, 150],
                    borderColor: '#10b981',
                    backgroundColor: 'rgba(16, 185, 129, 0.1)',
                    borderWidth: 3,
                    tension: 0.4,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { color: 'rgba(0,0,0,0.05)' }
                    },
                    x: {
                        grid: { display: false }
                    }
                }
            }
        });
    }
    </script>
</body>
</html>
